/**
 * Copyright (c) 2016, Uber Technologies, Inc
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package io.yarpc;


import io.yarpc.config.Config;
import io.yarpc.exception.OutboundNotFoundException;
import io.yarpc.handler.HandlerWrapper;
import io.yarpc.transport.Inbound;
import io.yarpc.transport.Outbound;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;

public class RPC {
    private static final Logger logger = LoggerFactory.getLogger(RPC.class);
    private Config config;
    private HandlerRegistry mHandlerRegistry;
    private RPCFuture runningFuture;

    public RPC(Config config) {
        this.mHandlerRegistry = new MapHandlerRegistry();
        this.config = config;
    }

    public void register(String proceduer, HandlerWrapper handler) {
        this.mHandlerRegistry.addHandler(proceduer, handler);
    }

    /**
     * Start all the registered inbounds servers.
     *
     * @return RPCFuture instance.
     * Caller should wait the future in order to prevent the main thread to exit.
     * The future will be resolved, when caller calls {@link #stop()},
     */
    public RPCFuture<Void> start() {
        this.runningFuture = new RPCFuture<Void>();
        List<Inbound> inbounds = this.config.getInbounds();
        for (Inbound inbound : inbounds) {
            try {
                inbound.setHandlerRegistry(this.mHandlerRegistry);
                inbound.start();
            } catch (Exception e) {
                this.runningFuture.setException(e);
            }
        }
        return this.runningFuture;
    }

    public Outbound lookUpOutbound(String service) throws OutboundNotFoundException {
        Outbound outbound = this.config.getOutbounds().get(service);
        if (outbound == null) {
            throw new OutboundNotFoundException(String.format("Cannot find outbound by service name: %s", service));
        }
        return outbound;
    }

    /**
     * Stop all the registered inbound servers.
     * The future returned by {@link #start()} will be resolved to null.
     */
    public void stop() {
        if (this.runningFuture == null || this.runningFuture.isCancelled() || this.runningFuture.isDone()) {
            return;
        }

        List<Inbound> inbounds = this.config.getInbounds();
        for (Inbound inbound : inbounds) {
            try {
                inbound.stop();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        this.runningFuture.set(null);
    }
}
