/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package x7.config;

import x7.core.config.Configs;
import x7.core.util.KeyUtil;
import x7.core.util.StringUtil;

import java.io.*;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.concurrent.ConcurrentHashMap;



public class TextParser{
	
	private static TextParser instance = null;
	private  Map<String, Object> map = null;
	
	public static TextParser getInstance(){
		if (instance == null){
			instance = new TextParser();
		}
		return instance;
	}
	private TextParser(){

	}
	
	
	public void load(String localAddress, String configSpace){
		
		map = Configs.referMap(configSpace);
		
		try{

			readConfigs(localAddress + "/"+ configSpace, configSpace);

		}catch (Exception e){
			e.printStackTrace();
			String notes = "无法启动";
			System.err.println("\n"+notes+"\n");			
			String err = "加载配置文件出错,请检查配置文件config/*.txt";
			System.err.println(err + "\n");
			try {
				Thread.sleep(20000);
			} catch (InterruptedException e1) {
				e1.printStackTrace();
			}
			System.exit(0);

		}
	}
	
	 
	
	public int getIntValue(String key){
	Integer value = 0;
	
		try{
			value = Integer.valueOf(map.get(key)+""); 
		}catch (MissingResourceException mre){
			String err = "请检查配置文件config/*.txt, 缺少key:" + key;
			System.err.println(err);
			mre.printStackTrace();
		}catch (Exception e){
			String err = "请检查配置文件config/*.txt, 发现了:" + key + "="+map.get(key);
			System.err.println(err);
			e.printStackTrace();
		}
		return value;
	}
	
	public String getString(String key){
		String value = "";
		
		try{
			value = map.get(key)+"";
		}catch (MissingResourceException mre){
			String err = "请检查配置文件config/*.txt, 缺少key:" + key;
			System.err.println(err);
			mre.printStackTrace();

		}catch (Exception e){
			String err = "请检查配置文件config/*.txt, 发现了:" + key + "="+map.get(key);
			System.err.println(err);
			e.printStackTrace();

		}
		return value;
	}
	
	public long getLongValue(String key){
		Long value = 0L;
		
		try{
			value = Long.valueOf(map.get(key)+"");
		}catch (MissingResourceException mre){
			String err = "请检查配置文件config/*.txt, 缺少key:" + key;
			System.err.println(err);
			mre.printStackTrace();

		}catch (Exception e){
			String err = "请检查配置文件config/*.txt, 发现了:" + key + "="+map.get(key);
			System.err.println(err);
			e.printStackTrace();
	
		}
		return value;
	}
	
	
	public void readConfigs(String path, String space){
		
		File file = new File(path);
		if (!file.isDirectory() || file.list().length == 0){
			throw new RuntimeException("CONFIG KEY EXCEPTION: x7.config.localAddress, if relative path unavailable, try absolute path");
		}
		if (file.isDirectory()){
			for (String childStr : file.list()){
				if (childStr.endsWith(".txt") || childStr.endsWith(".properties") || childStr.endsWith(".cfg") || childStr.endsWith(".init")
						|| childStr.endsWith(".js")
						|| childStr.endsWith(".json")){
					System.out.println("\n[" + space + "/" + childStr + "]");
					readConfig(path+"/"+childStr, space);
				}else if (! childStr.contains(".")){
					if (space == null || space.equals("")){
						readConfigs(path+"/"+childStr, childStr);
					}else{
						readConfigs(path+"/"+childStr, space+"."+childStr);
					}
				}
			}
			
		}
	}
	 
	/**
	 × 读取文件存入configData中
	 * @param path
	 * @return
	 */
	public void readConfig(String path,String space){
		FileInputStream fis=null;
		BufferedReader br=null;
		try {
			fis=new FileInputStream(path);
			br=new BufferedReader(new InputStreamReader(fis,"utf-8"));
			String dataStr="";

			String key = null;
			String value = null;
			
			while((dataStr=br.readLine())!=null){
				
				if (dataStr.startsWith("#"))
					continue;
				if(dataStr.contains("=")){
					
					put (key, value, space);
					key = null;
					value = null;
					
					//等号左边为key，等号右边为value
					key=dataStr.substring(0,dataStr.indexOf("=")).trim();
					value=dataStr.substring(dataStr.indexOf("=")+1);
						
					
				}else{
					if (StringUtil.isNullOrEmpty(dataStr))
						continue;
					if (dataStr.equals("\n"))
						continue;
					value += "\n";
					value += dataStr;
				}
				
			}
			
			put (key, value, space);
			key = null;
			value = null;
		} catch (Exception e) {
			e.printStackTrace();
		}finally{
			try {
				br.close();
				fis.close();
			} catch (IOException e) { 
				e.printStackTrace();
			}
		}
	}
	
	
	private void put(String key, String value, String space) {
		if (key!=null){
			System.out.println(key + "=" + value);
			if (space != null && !space.equals("")){
				key = space + "." + key;
			}
			
			value = value.trim();
			
			if (key.contains(".")){
				List<String> keyList = KeyUtil.getKeyList(key);
				int size = keyList.size();

				Map<String, Object> mapObject = map;
				int length = size - 1;
				for (int i = 0; i < length; i++) {
					String k = keyList.get(i);
					Object o = mapObject.get(k);
					if (o == null){
						o = new ConcurrentHashMap<String,Object>();
						mapObject.put(k, o);
					}
					mapObject = (Map<String, Object>) o;
				}
				mapObject.put(keyList.get(length), value);
			}else {
				map.put(key, value);
			}
		}
		
	}
	
	/**
	 * 将文件读到list中
	 * @param path
	 * @return
	 */
	public static List<String> readContent(String path){
		List<String> list=new ArrayList<String>();
		FileInputStream fis=null;
		BufferedReader br=null;
		try {
			fis=new FileInputStream(path);
			br=new BufferedReader(new InputStreamReader(fis,"utf-8"));
			String dataStr="";
			while((dataStr=br.readLine())!=null){ 
				list.add(dataStr);
			}
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}finally{
			try {
				br.close();
				fis.close();
			} catch (IOException e) { 
				e.printStackTrace();
			}
		}
		return list;
	}
	
	 
	/**
	 * 写入文件
	 * @param path
	 * @param data
	 */
	public static void writeConfig(String path,List<String> data){
		FileOutputStream fos=null;
		BufferedWriter br=null;
		try {
			fos=new FileOutputStream(path);
			br=new BufferedWriter(new OutputStreamWriter(fos,"utf-8"));
			for (String str : data) {
				br.write(str);
				br.newLine();
			}
		} catch (Exception e) { 
			e.printStackTrace();
		}finally{
			 try {
				br.flush();
				fos.flush();
				br.close();
				fos.close();
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	}


}
