package io.whitfin.elasticsearch.bulk;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import javax.annotation.Generated;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.Immutable;
import javax.annotation.concurrent.NotThreadSafe;

/**
 * Immutable implementation of {@link BulkOperation}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableBulkOperation.builder()}.
 */
@SuppressWarnings({"all"})
@ParametersAreNonnullByDefault
@Generated({"Immutables.generator", "BulkOperation"})
@Immutable
final class ImmutableBulkOperation extends BulkOperation {
  private final List<BulkAction> actions;
  private final long estimatedSizeInBytes;
  private final long numberOfActions;
  private final String payload;

  private ImmutableBulkOperation(ImmutableBulkOperation.Builder builder) {
    this.actions = createUnmodifiableList(true, builder.actions);
    this.estimatedSizeInBytes = initShim.estimatedSizeInBytes();
    this.numberOfActions = initShim.numberOfActions();
    this.payload = initShim.payload();
    this.initShim = null;
  }

  private static final int STAGE_INITIALIZING = -1;
  private static final int STAGE_UNINITIALIZED = 0;
  private static final int STAGE_INITIALIZED = 1;
  private transient volatile InitShim initShim = new InitShim();

  private final class InitShim {
    private long estimatedSizeInBytes;
    private int estimatedSizeInBytesBuildStage;

    long estimatedSizeInBytes() {
      if (estimatedSizeInBytesBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (estimatedSizeInBytesBuildStage == STAGE_UNINITIALIZED) {
        estimatedSizeInBytesBuildStage = STAGE_INITIALIZING;
        this.estimatedSizeInBytes = ImmutableBulkOperation.super.estimatedSizeInBytes();
        estimatedSizeInBytesBuildStage = STAGE_INITIALIZED;
      }
      return this.estimatedSizeInBytes;
    }
    private long numberOfActions;
    private int numberOfActionsBuildStage;

    long numberOfActions() {
      if (numberOfActionsBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (numberOfActionsBuildStage == STAGE_UNINITIALIZED) {
        numberOfActionsBuildStage = STAGE_INITIALIZING;
        this.numberOfActions = ImmutableBulkOperation.super.numberOfActions();
        numberOfActionsBuildStage = STAGE_INITIALIZED;
      }
      return this.numberOfActions;
    }
    private String payload;
    private int payloadBuildStage;

    String payload() {
      if (payloadBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (payloadBuildStage == STAGE_UNINITIALIZED) {
        payloadBuildStage = STAGE_INITIALIZING;
        this.payload = Objects.requireNonNull(ImmutableBulkOperation.super.payload(), "payload");
        payloadBuildStage = STAGE_INITIALIZED;
      }
      return this.payload;
    }

    private String formatInitCycleMessage() {
      ArrayList<String> attributes = new ArrayList<String>();
      if (estimatedSizeInBytesBuildStage == STAGE_INITIALIZING) attributes.add("estimatedSizeInBytes");
      if (numberOfActionsBuildStage == STAGE_INITIALIZING) attributes.add("numberOfActions");
      if (payloadBuildStage == STAGE_INITIALIZING) attributes.add("payload");
      return "Cannot build BulkOperation, attribute initializers form cycle" + attributes;
    }
  }

  /**
   * @return The value of the {@code actions} attribute
   */
  @JsonProperty("actions")
  @Override
  public List<BulkAction> actions() {
    return actions;
  }

  /**
   * @return The computed-at-construction value of the {@code estimatedSizeInBytes} attribute
   */
  @JsonProperty("estimatedSizeInBytes")
  @Override
  public long estimatedSizeInBytes() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.estimatedSizeInBytes()
        : this.estimatedSizeInBytes;
  }

  /**
   * @return The computed-at-construction value of the {@code numberOfActions} attribute
   */
  @JsonProperty("numberOfActions")
  @Override
  public long numberOfActions() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.numberOfActions()
        : this.numberOfActions;
  }

  /**
   * @return The computed-at-construction value of the {@code payload} attribute
   */
  @JsonProperty("payload")
  @Override
  public String payload() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.payload()
        : this.payload;
  }

  /**
   * This instance is equal to all instances of {@code ImmutableBulkOperation} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(@Nullable Object another) {
    if (this == another) return true;
    return another instanceof ImmutableBulkOperation
        && equalTo((ImmutableBulkOperation) another);
  }

  private boolean equalTo(ImmutableBulkOperation another) {
    return actions.equals(another.actions)
        && estimatedSizeInBytes == another.estimatedSizeInBytes
        && numberOfActions == another.numberOfActions
        && payload.equals(another.payload);
  }

  /**
   * Computes a hash code from attributes: {@code actions}, {@code estimatedSizeInBytes}, {@code numberOfActions}, {@code payload}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + actions.hashCode();
    h += (h << 5) + (int) (estimatedSizeInBytes ^ (estimatedSizeInBytes >>> 32));
    h += (h << 5) + (int) (numberOfActions ^ (numberOfActions >>> 32));
    h += (h << 5) + payload.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code BulkOperation} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "BulkOperation{"
        + "actions=" + actions
        + ", estimatedSizeInBytes=" + estimatedSizeInBytes
        + ", numberOfActions=" + numberOfActions
        + ", payload=" + payload
        + "}";
  }

  /**
   * Utility type used to correctly read immutable object from JSON representation.
   * @deprecated Do not use this type directly, it exists only for the <em>Jackson</em>-binding infrastructure
   */
  @Deprecated
  @JsonDeserialize
  @JsonAutoDetect(fieldVisibility = JsonAutoDetect.Visibility.NONE)
  static final class Json extends BulkOperation {
    @Nullable List<BulkAction> actions = Collections.emptyList();
    @JsonProperty("actions")
    public void setActions(List<BulkAction> actions) {
      this.actions = actions;
    }
    @Override
    public List<BulkAction> actions() { throw new UnsupportedOperationException(); }
    @Override
    public long estimatedSizeInBytes() { throw new UnsupportedOperationException(); }
    @Override
    public long numberOfActions() { throw new UnsupportedOperationException(); }
    @Override
    public String payload() { throw new UnsupportedOperationException(); }
  }

  /**
   * @param json A JSON-bindable data structure
   * @return An immutable value type
   * @deprecated Do not use this method directly, it exists only for the <em>Jackson</em>-binding infrastructure
   */
  @Deprecated
  @JsonCreator(mode = JsonCreator.Mode.DELEGATING)
  static ImmutableBulkOperation fromJson(Json json) {
    ImmutableBulkOperation.Builder builder = ImmutableBulkOperation.builder();
    if (json.actions != null) {
      builder.addAllActions(json.actions);
    }
    return builder.build();
  }

  /**
   * Creates a builder for {@link ImmutableBulkOperation ImmutableBulkOperation}.
   * @return A new ImmutableBulkOperation builder
   */
  public static ImmutableBulkOperation.Builder builder() {
    return new ImmutableBulkOperation.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableBulkOperation ImmutableBulkOperation}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  @NotThreadSafe
  public static final class Builder 
      implements BulkOperation.Builder {
    private List<BulkAction> actions = new ArrayList<BulkAction>();

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code BulkOperation} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(BulkOperation instance) {
      Objects.requireNonNull(instance, "instance");
      addAllActions(instance.actions());
      return this;
    }

    /**
     * Adds one element to {@link BulkOperation#actions() actions} list.
     * @param element A actions element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAction(BulkAction element) {
      this.actions.add(Objects.requireNonNull(element, "actions element"));
      return this;
    }

    /**
     * Adds elements to {@link BulkOperation#actions() actions} list.
     * @param elements An array of actions elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAction(BulkAction... elements) {
      for (BulkAction element : elements) {
        this.actions.add(Objects.requireNonNull(element, "actions element"));
      }
      return this;
    }

    /**
     * Sets or replaces all elements for {@link BulkOperation#actions() actions} list.
     * @param elements An iterable of actions elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("actions")
    public final Builder actions(Iterable<? extends BulkAction> elements) {
      this.actions.clear();
      return addAllActions(elements);
    }

    /**
     * Adds elements to {@link BulkOperation#actions() actions} list.
     * @param elements An iterable of actions elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllActions(Iterable<? extends BulkAction> elements) {
      for (BulkAction element : elements) {
        this.actions.add(Objects.requireNonNull(element, "actions element"));
      }
      return this;
    }

    /**
     * Builds a new {@link ImmutableBulkOperation ImmutableBulkOperation}.
     * @return An immutable instance of BulkOperation
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableBulkOperation build() {
      return new ImmutableBulkOperation(this);
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<T>();
    } else {
      list = new ArrayList<T>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<T>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }
}
