/*
 * Decompiled with CFR 0.152.
 */
package io.timson.firehose.stream;

import com.amazonaws.services.kinesisfirehose.model.CompressionFormat;
import io.timson.firehose.aws.S3Client;
import io.timson.firehose.stream.DeliveryStream;
import io.timson.firehose.util.FirehoseUtil;
import java.io.IOException;
import java.nio.charset.Charset;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.Timer;
import java.util.TimerTask;
import java.util.UUID;
import java.util.logging.Level;
import java.util.logging.Logger;

public class S3DeliveryStream
implements DeliveryStream {
    private static final Logger logger = Logger.getLogger(S3DeliveryStream.class.getName());
    private static final Charset UTF_8 = Charset.forName("UTF-8");
    private static final long MEGABYTE = 0x100000L;
    private static final int VERSION = 1;
    private static final DateTimeFormatter YYYY_MM_DD_HH = DateTimeFormatter.ofPattern("yyyy/MM/dd/HH");
    private static final DateTimeFormatter YYYY_MM_DD_HH_MM_SS = DateTimeFormatter.ofPattern("yyyy-MM-dd-HH-mm-ss");
    private final String name;
    private final S3Client s3Client;
    private final String s3Bucket;
    private final String s3Prefix;
    private final Long bufferIntervalMs;
    private final Long bufferFlushSizeBytes;
    private final CompressionFormat compressionFormat;
    private StringBuilder buffer = new StringBuilder();
    private long bufferSize = 0L;
    private TimerTask flushTimerTask;
    private Timer flushTimer;

    private S3DeliveryStream(String name, S3Client s3Client, String s3BucketArn, String s3Prefix, Long bufferIntervalMs, Long bufferSizeBytes, CompressionFormat compressionFormat) {
        this.name = name;
        this.s3Client = s3Client;
        this.s3Bucket = this.extractBucketName(s3BucketArn);
        this.s3Prefix = s3Prefix;
        this.bufferIntervalMs = bufferIntervalMs;
        this.bufferFlushSizeBytes = bufferSizeBytes;
        this.compressionFormat = compressionFormat;
    }

    private String extractBucketName(String s3BucketArn) {
        if (!s3BucketArn.startsWith("arn:aws:s3:::")) {
            throw new IllegalArgumentException(String.format("Invalid Bucket ARN %s", s3BucketArn));
        }
        return s3BucketArn.substring("arn:aws:s3:::".length());
    }

    private void startFlushTimer() {
        this.flushTimerTask = new TimerTask(){

            @Override
            public void run() {
                S3DeliveryStream.this.flush();
            }
        };
        this.flushTimer = new Timer();
        this.flushTimer.schedule(this.flushTimerTask, this.bufferIntervalMs);
    }

    private void stopFlushTimer() {
        if (this.flushTimer == null) {
            return;
        }
        this.flushTimer.cancel();
    }

    @Override
    public synchronized void write(String data) {
        if (this.bufferSize == 0L) {
            this.startFlushTimer();
        }
        this.buffer.append(data);
        this.bufferSize += (long)data.getBytes(UTF_8).length;
        if (this.bufferSize >= this.bufferFlushSizeBytes) {
            this.stopFlushTimer();
            this.flush();
        }
    }

    @Override
    public void stop() {
        this.stopFlushTimer();
    }

    @Override
    public String getName() {
        return this.name;
    }

    private synchronized void flush() {
        if (this.bufferSize > 0L) {
            String s3Path = this.generateS3Path();
            try {
                this.createS3Object(this.s3Bucket, s3Path, this.buffer.toString());
                this.bufferSize = 0L;
                this.buffer.setLength(0);
            }
            catch (IOException e) {
                logger.log(Level.SEVERE, String.format("Unable to save S3 Object s3://%s/%s", this.s3Bucket, s3Path), e);
            }
        }
    }

    private void createS3Object(String s3Bucket, String s3Path, String content) throws IOException {
        switch (this.compressionFormat) {
            case Snappy: {
                this.s3Client.createSnappyObject(s3Bucket, s3Path, content);
                break;
            }
            case GZIP: {
                this.s3Client.createGzipObject(s3Bucket, s3Path, content);
                break;
            }
            case ZIP: {
                this.s3Client.createZipObject(s3Bucket, s3Path, content);
                break;
            }
            default: {
                this.s3Client.createObject(s3Bucket, s3Path, content);
            }
        }
    }

    private String generateS3Path() {
        LocalDateTime now = LocalDateTime.now(ZoneOffset.UTC);
        return this.s3Prefix + now.format(YYYY_MM_DD_HH) + '/' + this.name + '-' + 1 + '-' + now.format(YYYY_MM_DD_HH_MM_SS) + '-' + UUID.randomUUID().toString();
    }

    public static class S3DeliveryStreamBuilder {
        private String name;
        private S3Client s3Client;
        private String s3BucketArn;
        private String s3Prefix = "";
        private Long bufferIntervalMs = 300000L;
        private Long bufferSizeBytes = 0x500000L;
        private CompressionFormat compressionFormat = CompressionFormat.UNCOMPRESSED;

        public S3DeliveryStreamBuilder withName(String name) {
            this.name = name;
            return this;
        }

        public S3DeliveryStreamBuilder withS3Client(S3Client s3Client) {
            this.s3Client = s3Client;
            return this;
        }

        public S3DeliveryStreamBuilder withS3BucketArn(String s3BucketArn) {
            this.s3BucketArn = s3BucketArn;
            return this;
        }

        public S3DeliveryStreamBuilder withS3Prefix(String s3Prefix) {
            if (s3Prefix == null) {
                return this;
            }
            this.s3Prefix = s3Prefix;
            return this;
        }

        public S3DeliveryStreamBuilder withBufferIntervalSeconds(Integer bufferIntervalSeconds) {
            if (bufferIntervalSeconds == null) {
                return this;
            }
            this.bufferIntervalMs = (long)bufferIntervalSeconds.intValue() * 1000L;
            return this;
        }

        public S3DeliveryStreamBuilder withBufferIntervalMilliseconds(Long bufferIntervalMilliseconds) {
            if (bufferIntervalMilliseconds == null) {
                return this;
            }
            this.bufferIntervalMs = bufferIntervalMilliseconds;
            return this;
        }

        public S3DeliveryStreamBuilder withBufferSizeMB(Integer bufferSizeMb) {
            if (bufferSizeMb == null) {
                return this;
            }
            this.bufferSizeBytes = (long)bufferSizeMb.intValue() * 0x100000L;
            return this;
        }

        public S3DeliveryStreamBuilder withBufferSizeBytes(Long bufferSizeBytes) {
            if (bufferSizeBytes == null) {
                return this;
            }
            this.bufferSizeBytes = bufferSizeBytes;
            return this;
        }

        public S3DeliveryStreamBuilder withCompressionFormat(CompressionFormat compressionFormat) {
            if (compressionFormat == null) {
                return this;
            }
            this.compressionFormat = compressionFormat;
            return this;
        }

        public S3DeliveryStream build() {
            if (FirehoseUtil.isEmpty(this.name)) {
                throw new IllegalArgumentException("Delivery stream name cannot be empty");
            }
            if (FirehoseUtil.isEmpty(this.s3BucketArn)) {
                throw new IllegalArgumentException("S3 Bucket ARN cannot be empty");
            }
            return new S3DeliveryStream(this.name, this.s3Client, this.s3BucketArn, this.s3Prefix, this.bufferIntervalMs, this.bufferSizeBytes, this.compressionFormat);
        }
    }
}

