package com.mcarthurlabs.result

/**
 * ResultApp
 * Created by matt on 11/28/18.
 */

sealed class Result<T> {
    data class Success<T>(val value: T) : Result<T>()
    data class Failure<T>(val error: Exception) : Result<T>()

    fun value(): T? {
        return when (this) {
            is Success -> this.value
            is Failure -> null
        }
    }

    fun value(default: T): T {
        return when (this) {
            is Success -> this.value
            is Failure -> default
        }
    }

    fun error(): Exception? {
        return when (this) {
            is Success -> null
            is Failure -> this.error
        }
    }

    fun <U>map(function: (T) -> U): Result<U> {
        return when(this){
            is Success -> Result.Success(function(this.value))
            is Failure -> Result.Failure(this.error)
        }
    }

    fun <U>flatMap(function: (T) -> Result<U>): Result<U> {
        return when(this){
            is Success -> function(this.value)
            is Failure -> Result.Failure(this.error)
        }
    }

    fun <U>tryMap(function: (T) -> U): Result<U> {
        return flatMap { value: T ->
            try {
              val newValue = function(value)
                Result.Success(newValue)
            }catch(error: Exception) {
                Result.Failure<U>(error)
            }
        }
    }
}

typealias ResultCompletion<T> = (Result<T>) -> Unit