package io.teliver.sdk.util;

import android.Manifest;
import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Intent;
import android.content.IntentSender;
import android.content.pm.PackageManager;
import android.location.Location;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;
import android.view.View;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.app.ActionBar;
import androidx.appcompat.app.AppCompatActivity;
import androidx.appcompat.widget.Toolbar;
import androidx.core.app.ActivityCompat;
import androidx.core.content.ContextCompat;

import com.google.android.gms.common.api.ResolvableApiException;
import com.google.android.gms.location.FusedLocationProviderClient;
import com.google.android.gms.location.LocationCallback;
import com.google.android.gms.location.LocationRequest;
import com.google.android.gms.location.LocationResult;
import com.google.android.gms.location.LocationServices;
import com.google.android.gms.location.LocationSettingsRequest;
import com.google.android.gms.location.LocationSettingsResponse;
import com.google.android.gms.maps.CameraUpdateFactory;
import com.google.android.gms.maps.GoogleMap;
import com.google.android.gms.maps.OnMapReadyCallback;
import com.google.android.gms.maps.SupportMapFragment;
import com.google.android.gms.maps.model.BitmapDescriptorFactory;
import com.google.android.gms.maps.model.LatLng;
import com.google.android.gms.maps.model.Marker;
import com.google.android.gms.maps.model.MarkerOptions;
import com.google.android.gms.tasks.OnFailureListener;
import com.google.android.gms.tasks.OnSuccessListener;
import com.google.android.gms.tasks.Task;
import io.teliver.sdk.R;
import io.teliver.sdk.core.TLog;
import io.teliver.sdk.core.TrackingListener;
import io.teliver.sdk.models.MarkerOption;
import io.teliver.sdk.models.TConstants;
import io.teliver.sdk.models.TLocation;
import io.teliver.sdk.models.TrackingOptions;

import java.util.HashMap;

@SuppressWarnings({"MissingPermission", "ConstantConditions"})
public class TeliverMap extends AppCompatActivity implements OnMapReadyCallback,
        View.OnClickListener, TConstants, TrackingListener, OnSuccessListener<Location> {

    private GoogleMap googleMap;

    private FusedLocationProviderClient providerClient;

    private HashMap<String, Marker> markers;

    private TrackingOptions trackingOptions;

    private Tracker tracker;

    @SuppressLint("StaticFieldLeak")
    public static Activity mapActivity;

    private String permission = Manifest.permission.ACCESS_FINE_LOCATION;

    @Override
    protected void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.map_view);
    }

    @Override
    protected void onPostCreate(@Nullable Bundle savedInstanceState) {
        super.onPostCreate(savedInstanceState);
        mapActivity = this;
        SupportMapFragment mapFragment = (SupportMapFragment) getSupportFragmentManager().findFragmentById(R.id.mapView);
        mapFragment.getMapAsync(this);
        Toolbar toolbar = findViewById(R.id.toolbar);
        setSupportActionBar(toolbar);
        ActionBar actionBar = getSupportActionBar();
        actionBar.setDisplayHomeAsUpEnabled(true);
        actionBar.setDisplayShowHomeEnabled(true);
        trackingOptions = getIntent().getParcelableExtra(TConstants.DOTS_OBJ);
        String title = trackingOptions.getPageTitle();
        actionBar.setTitle(title.isEmpty() ? getString(R.string.lib_app_name) : title);
        toolbar.setNavigationOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                finish();
            }
        });
        findViewById(R.id.btn_user_gps).setOnClickListener(this);
        markers = new HashMap<>();
        trackingOptions.setTrackingListener(this);
        tracker = Tracker.getInstance(this);
        tracker.startTracking(trackingOptions);
    }

    @Override
    public void onMapReady(GoogleMap map) {
        try {
            this.googleMap = map;
            providerClient = LocationServices.getFusedLocationProviderClient(this);
            googleMap.setMapType(GoogleMap.MAP_TYPE_TERRAIN);
            googleMap.getUiSettings().setMyLocationButtonEnabled(false);
            googleMap.getUiSettings().setMapToolbarEnabled(false);
            googleMap.getUiSettings().setCompassEnabled(false);
            if (isPermissionGranted())
                enableGpsLocation();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void enableGpsLocation() {
        try {
            final LocationRequest locationRequest = LocationRequest.create();
            locationRequest.setPriority(LocationRequest.PRIORITY_HIGH_ACCURACY);
            locationRequest.setInterval(2000);
            locationRequest.setFastestInterval(3000);
            LocationSettingsRequest request = new LocationSettingsRequest
                    .Builder().addLocationRequest(locationRequest)
                    .setAlwaysShow(true).build();
            Task<LocationSettingsResponse> task = LocationServices.getSettingsClient(
                    this).checkLocationSettings(request);
            task.addOnSuccessListener(new OnSuccessListener<LocationSettingsResponse>() {
                @Override
                public void onSuccess(LocationSettingsResponse locationSettingsResponse) {
                    getMyLocation(locationRequest);
                }
            });
            task.addOnFailureListener(new OnFailureListener() {
                @Override
                public void onFailure(@NonNull Exception e) {
                    if (e instanceof ResolvableApiException) {
                        try {
                            ResolvableApiException resolvable = (ResolvableApiException) e;
                            resolvable.startResolutionForResult(TeliverMap.this, 110);
                        } catch (IntentSender.SendIntentException sendEx) {
                            Log.e("ulka:", "gps error::");
                        }
                    }
                }
            });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void getMyLocation(LocationRequest locationRequest) {
        try {
            providerClient.requestLocationUpdates(locationRequest, new LocationCallback() {
                @Override
                public void onLocationResult(LocationResult locationResult) {
                    if (locationResult == null)
                        onSuccess(null);
                    else {
                        Location location = locationResult.getLastLocation();
                        providerClient.removeLocationUpdates(this);
                        onSuccess(location);
                    }
                }
            }, Looper.myLooper());
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public void onClick(View view) {
        if (view.getId() == R.id.btn_user_gps && checkPermission())
            enableGpsLocation();
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, @NonNull String[] permissions
            , @NonNull int[] grantResults) {
        if (requestCode == PERMISSION_REQ_CODE && grantResults[0] == PackageManager.PERMISSION_GRANTED)
            enableGpsLocation();
    }

    @Override
    public void onActivityResult(int requestCode, int resultCode, Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        if (Activity.RESULT_OK == resultCode && 110 == requestCode)
            enableGpsLocation();
    }

    @Override
    public void onTrackingStarted(String trackingID) {
        TLog.log("TMap: Tracking Started:::" + trackingID);
    }

    @Override
    public void onLocationUpdate(String trackingId, TLocation location) {
        if (markers.containsKey(trackingId))
            TUtils.animateMarker(googleMap, markers.get(trackingId), location);
        else {
            for (MarkerOption markerOption : trackingOptions.getMarkerOptions()) {
                if (!trackingId.equals(markerOption.getTrackingId()))
                    continue;
                LatLng latLng = new LatLng(location.getLatitude(), location.getLongitude());
                MarkerOptions options = new MarkerOptions().position(latLng);
                if (markerOption.getBitmap() != null)
                    options.icon(BitmapDescriptorFactory.fromBitmap(markerOption.getBitmap()));
                else
                    options.icon(BitmapDescriptorFactory.fromResource(markerOption.getIconMarker()));
                Marker marker = googleMap.addMarker(options);
                marker.setTitle(markerOption.getMarkerTitle());
                marker.setSnippet(markerOption.getMarkerSnippet());
                markers.put(trackingId, marker);
            }
        }

        new Handler().postDelayed(new Runnable() {
            @Override
            public void run() {
                TUtils.maintainCameraPosition(TeliverMap.this, markers, googleMap);
            }
        }, 700);
    }

    @Override
    public void onTrackingEnded(String trackingId) {
        try {
            if (markers.containsKey(trackingId)) {
                Marker marker = markers.get(trackingId);
                if (marker != null)
                    marker.remove();
                markers.remove(trackingId);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public void onTrackingError(String reason) {
        TLog.log("TMap: Tracking Error:::" + reason);
    }

    @Override
    public void onBackPressed() {
        disconnect();
        super.onBackPressed();
    }

    private boolean checkPermission() {
        if (isPermissionGranted())
            return true;
        if (ActivityCompat.shouldShowRequestPermissionRationale(this,
                permission))
            ActivityCompat.requestPermissions(this,
                    new String[]{permission}, TConstants.PERMISSION_REQ_CODE);
        else
            ActivityCompat.requestPermissions(this,
                    new String[]{permission}, TConstants.PERMISSION_REQ_CODE);
        return false;
    }

    private boolean isPermissionGranted() {
        return ContextCompat.checkSelfPermission(this, permission)
                == PackageManager.PERMISSION_GRANTED;
    }

    public void disconnect() {
        if (tracker != null && trackingOptions != null)
            tracker.disconnect(trackingOptions);
    }

    @Override
    public void onSuccess(Location location) {
        try {
            if (isPermissionGranted())
                googleMap.setMyLocationEnabled(true);
            if (location != null) {
                LatLng latLng = new LatLng(location.getLatitude(), location.getLongitude());
                googleMap.animateCamera(CameraUpdateFactory.newLatLngZoom(latLng, 16));
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}
