package io.teliver.sdk.util;

import android.annotation.SuppressLint;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.IBinder;
import android.widget.Toast;

import com.google.firebase.FirebaseApp;
import io.teliver.sdk.core.EventListener;
import io.teliver.sdk.core.InitStatusListener;
import io.teliver.sdk.core.TLog;
import io.teliver.sdk.core.TService;
import io.teliver.sdk.core.TaskListListener;
import io.teliver.sdk.core.TaskListener;
import io.teliver.sdk.core.TripListener;
import io.teliver.sdk.models.NotificationData;
import io.teliver.sdk.models.PushData;
import io.teliver.sdk.models.TConstants;
import io.teliver.sdk.models.TResponse;
import io.teliver.sdk.models.Task;
import io.teliver.sdk.models.TaskList;
import io.teliver.sdk.models.TaskOptions;
import io.teliver.sdk.models.TrackingOptions;
import io.teliver.sdk.models.Trip;
import io.teliver.sdk.models.TripOptions;
import io.teliver.sdk.models.User;

import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

import okhttp3.FormBody;
import okhttp3.MediaType;
import okhttp3.RequestBody;

import static io.teliver.sdk.util.TUtils.getGson;

public class SdkHandler {

    private TPreference preference;

    private Context context;

    private TService service;

    @SuppressLint("StaticFieldLeak")
    private static SdkHandler handler;

    private InitStatusListener initListener;

    private final MyServiceConnection serviceConnection = new MyServiceConnection();

    private final class MyServiceConnection implements ServiceConnection {

        @Override
        public void onServiceConnected(ComponentName name, IBinder binder) {
            service = ((ServiceBinder) binder).getService();
        }

        @Override
        public void onServiceDisconnected(ComponentName name) {
            service = null;
        }
    }

    public static SdkHandler getInstance(Context context) {
        if (handler == null)
            handler = new SdkHandler(context);
        return handler;
    }

    private SdkHandler(Context context) {
        this.context = context;
        preference = new TPreference(context);
        FirebaseApp.initializeApp(context);
        TUtils.saveToken(context);
    }

    public void initializeClient(String apiKey) {
        preference.storeString(TConstants.API_KEY, apiKey);
        if (TUtils.isNetNotConnected(context))
            TLog.log("Internet not Connected");
        else {
            TRestCall tRestCall = new TRestCall(context);
            tRestCall.setCallBackListener(new InitListener());
            FormBody.Builder builder = new FormBody.Builder();
            builder.add("device_details", TUtils.getDeviceInfo(context));
            builder.add("security", TUtils.getSecureInfo(context));
            tRestCall.requestApi("init", builder.build());
        }
    }

    public void unRegisterUser() {
        if (TUtils.isNetNotConnected(context))
            TLog.log("Internet is not Connected");
        else {
            TRestCall tRestCall = new TRestCall(context);
            FormBody.Builder builder = new FormBody.Builder();
            builder.add("user_id", preference.getUserId());
            tRestCall.requestApi("unregister_user", builder.build());
        }
    }

    public void identifyDeviceUser(User user) {
        if (user == null)
            TLog.log("User Object is Null");
        else if (user.getId().isEmpty())
            TLog.log("User ID should not be empty");
        else if (TUtils.isNetNotConnected(context))
            TLog.log("Internet is not Connected");
        else {
            preference.saveUserId(user.getId());
            TRestCall tRestCall = new TRestCall(context);
            FormBody.Builder builder = new FormBody.Builder();
            builder.add("user_type", user.getType());
            builder.add("user_id", user.getId());
            builder.add("device_details", TUtils.getDeviceInfo(context));
            builder.add("security", TUtils.getSecureInfo(context));
            builder.add("phone", user.getPhone());
            builder.add("email", user.getEmail());
            builder.add("name", user.getName());
            if (user.isRegisteredForPush())
                builder.add("push_key", preference.getPushToken());
            tRestCall.requestApi("identify_user", builder.build());
        }
    }

    public void startTripUpdates(TripOptions options) {
        if (options == null)
            TLog.log("Builder is Null");
        else if (options.getTrackingId().isEmpty())
            TLog.log("Please specify Trip Tracking ID");
        else if (TUtils.isNetNotConnected(context))
            TLog.log("Internet is not Connected");
        if (TUtils.isPermissionNotOk(context)) {
            String locationText = "Location Permission not Enabled. Please enable Permission";
            Toast.makeText(context, locationText, Toast.LENGTH_LONG).show();
            TLog.log(locationText);
        } else if (!TUtils.isGpsEnabled(context)) {
            String locationText = "Please enable GPS";
            Toast.makeText(context, locationText, Toast.LENGTH_LONG).show();
            TLog.log(locationText);
        } else if (service != null)
            service.prepareUpdates(options, preference.getUserId());
        else TLog.log("Service is null");
    }

    public void stopTripUpdates(String trackingId) {
        if (TUtils.clearNull(trackingId).isEmpty())
            TLog.log("Please provide Tracking id");
        else if (TUtils.isPermissionNotOk(context))
            TLog.log("Location Permission not Enabled. Please enable Permission");
        else if (TUtils.isNetNotConnected(context))
            TLog.log("Internet is not Connected");
        else if (service != null)
            service.stopTripUpdates(trackingId);
    }

    public void addTag(String trackingId, String tag) {
        if (TUtils.clearNull(trackingId).isEmpty())
            TLog.log("Please provide Tracking id");
        else if (TUtils.clearNull(tag).isEmpty())
            TLog.log("Please provide Tag");
        else if (service != null)
            service.tagLocation(trackingId, tag);
    }


    public void setListener(TripListener listener) {
        if (service != null && listener != null)
            service.updateListener(listener);
    }

    public List<Trip> getCurrentTrips() {
        if (service != null)
            return service.getCurrentTrips();
        return new ArrayList<>();
    }

    public void startTracking(TrackingOptions options) {
        if (checkBuilder(options))
            return;
        if (options.getMapObject() == null)
            context.startActivity(new Intent(context, TeliverMap.class)
                    .setFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
                    .putExtra(TConstants.DOTS_OBJ, options));
        else Tracker.getInstance(context).startTracking(options);
    }

    public void stopTracking(List<String> trackingIds) {
        if (trackingIds == null || trackingIds.isEmpty())
            TLog.log("Tracking ID should not be empty or null");
        else Tracker.getInstance(context).stopTracking(trackingIds);
    }

    public void setInitListener(InitStatusListener initListener) {
        this.initListener = initListener;
    }

    private class InitListener implements TRestCall.ResponseListener {
        @Override
        public void onResponse(String result) {
            TLog.log(result);
            try {
                if (result.isEmpty())
                    return;
                TResponse response = getGson().fromJson(result, TResponse.class);
                if (response.isSuccess()) {
                    preference.saveAuthToken(response.getToken());
                    startTService();
                    if (initListener != null)
                        initListener.onSuccess();
                } else {
                    TLog.log("On Init:Failure->Reason: " + response.getMessage());
                    if (initListener != null)
                        initListener.OnFailure(response.getMessage());
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    private boolean checkBuilder(TrackingOptions options) {
        if (options == null || options.getMarkerOptions().isEmpty())
            TLog.log("Marker Options should not be Empty");
        else if (TUtils.isNetNotConnected(context))
            TLog.log("Internet is not Connected");
        else return false;
        return true;
    }

    private void startTService() {
        Intent serviceStartIntent = new Intent(context, TService.class);
        context.startService(serviceStartIntent);
        context.bindService(serviceStartIntent, serviceConnection,
                Context.BIND_AUTO_CREATE);
    }

    public void sendEventPush(String trackingId, PushData pushData) {
        try {
            if (pushData == null)
                TLog.log("Push Data should not be null");
            else if (trackingId.isEmpty())
                TLog.log("Trip ID should not be null");
            else if (TUtils.isNetNotConnected(context))
                TLog.log("Internet is not Connected");
            else {
                NotificationData data = new NotificationData();
                data.setCommand(TConstants.CMD_EVENT_PUSH);
                data.setMessage(pushData.getMessage());
                data.setPayload(pushData.getPayload());
                data.setTrackingID(trackingId);
                RequestBody body = new FormBody.Builder()
                        .add(TConstants.PUSH_TITLE, pushData.getMessage())
                        .add(TConstants.PUSH_DATA, getGson().toJson(data))
                        .add("trip_id", trackingId)
                        .add(TConstants.PUSH_IDS, getGson().toJson(pushData.getUsers()))
                        .build();
                TRestCall tRestCall = new TRestCall(context);
                tRestCall.requestApi("send_event_push", body);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void closeMap() {
        try {
            if (TeliverMap.mapActivity != null) {
                Tracker.getInstance(context).disconnectClient();
                TeliverMap.mapActivity.finish();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void getMyTaskList(String status, int page, int limit, final TaskListListener listener) {
        if (preference.getUserId().isEmpty())
            listener.onFailure("Please identify this user with driver id to get assigned tasks.");
        else if (TUtils.isNetNotConnected(context))
            TLog.log("Internet is not Connected");
        else {
            TRestCall tRestCall = new TRestCall(context);
            tRestCall.setHttpType(TRestCall.HTTP_TYPE.TYPE_GET);
            tRestCall.setCallBackListener(new TRestCall.ResponseListener() {
                @Override
                public void onResponse(String result) {
                    try {
                        JSONObject jsonObject = new JSONObject(result);
                        if (jsonObject.getBoolean("success")) {
                            TaskList list = getGson().fromJson(jsonObject.getJSONObject("data").toString(), TaskList.class);
                            listener.onSuccess(list.getTaskList(), list.getTotal());
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            });
            tRestCall.requestApi("task/list?driver_id=" + preference.getUserId() + "&status=" + status + "&page=" + page + "&limit=" + limit, null);
        }
    }

    public void getMyTaskWithTaskId(String taskId, final TaskListener listener) {
        if (preference.getUserId().isEmpty())
            listener.onFailure("Please identify this user with driver id to get assigned tasks.");
        else if (TUtils.clearNull(taskId).isEmpty())
            listener.onFailure("Please provide a valid task Id.");
        else if (TUtils.isNetNotConnected(context))
            TLog.log("Internet is not Connected");
        else {
            TRestCall tRestCall = new TRestCall(context);
            tRestCall.setHttpType(TRestCall.HTTP_TYPE.TYPE_GET);
            tRestCall.setCallBackListener(new TRestCall.ResponseListener() {
                @Override
                public void onResponse(String result) {
                    try {
                        JSONObject jsonObject = new JSONObject(result);
                        if (jsonObject.getBoolean("success")) {
                            Task task = getGson().fromJson(jsonObject.getJSONObject("data")
                                    .getJSONObject("task").toString(), Task.class);
                            if (preference.getUserId().
                                    equalsIgnoreCase(task.getDriver().getId()))
                                listener.onSuccess(task);
                            else
                                listener.onFailure("This task is not for you");
                        } else
                            listener.onFailure("Some error occurred, try again later");
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            });
            tRestCall.requestApi("task?driver_id=" + preference.getUserId() + "&task_id=" + taskId, null);
        }
    }

    public void updateTaskStatus(String taskId, String status, final EventListener listener) {
        if (preference.getUserId().isEmpty())
            TLog.log("Please identify this user with driver id to update.");
        else if (TUtils.isNetNotConnected(context))
            TLog.log("Internet is not Connected");
        else {
            TRestCall tRestCall = new TRestCall(context);
            tRestCall.setHttpType(TRestCall.HTTP_TYPE.TYPE_PATCH);
            tRestCall.setCallBackListener(new TRestCall.ResponseListener() {
                @Override
                public void onResponse(String result) {
                    listener.onSuccess("Task Updated");
                }
            });
            FormBody.Builder builder = new FormBody.Builder();
            builder.add("status", status);
            tRestCall.requestApi("task?task_id=" + taskId, builder.build());
        }
    }

    public void updateAvailability(final boolean status, final EventListener listener) {
        TRestCall tRestCall = new TRestCall(context);
        tRestCall.setHttpType(TRestCall.HTTP_TYPE.TYPE_PATCH);
        tRestCall.setCallBackListener(new TRestCall.ResponseListener() {
            @Override
            public void onResponse(String result) {
                try {
                    JSONObject jsonObject = new JSONObject(result);
                    if (service == null)
                        TLog.log("Service is null");
                    else if (jsonObject.getBoolean("success")) {
                        service.updateAvailability(status);
                        if (listener != null)
                            listener.onSuccess("User updated successfully");
                    } else if (listener != null)
                        listener.onFailure("Unable to update user try again later");
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        });
        FormBody.Builder builder = new FormBody.Builder();
        builder.add("availability", String.valueOf(status));
        tRestCall.requestApi("update_user?driver_id=" + preference.getUserId(), builder.build());
    }

    public void getMyTask(final TaskOptions taskOptions) {
        if (TUtils.isNetNotConnected(context))
            TLog.log("Internet is not Connected");
        else {
            String startDate = taskOptions.getStartDate();
            String endDate = taskOptions.getEndDate();
            int page = taskOptions.getPage();
            final int limit = taskOptions.getLimit();
            String status = taskOptions.getStatus();
            TRestCall tRestCall = new TRestCall(context);
            tRestCall.setHttpType(TRestCall.HTTP_TYPE.TYPE_GET);
            tRestCall.setCallBackListener(new TRestCall.ResponseListener() {
                @Override
                public void onResponse(String result) {
                    try {
                        JSONObject jsonObject = new JSONObject(result);
                        if (jsonObject.getBoolean("success")) {
                            TaskList list = getGson().fromJson(jsonObject.getJSONObject("data").toString(), TaskList.class);
                            taskOptions.getTaskListener().onSuccess(list.getTaskList(), list.getTotal());
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            });
            tRestCall.requestApi("task/list?driver_id=" + preference.getUserId()
                            + (!startDate.isEmpty() ? "&from_time=" + startDate : "")
                            + (!status.isEmpty() ? "&status=" + status : "")
                            + (page > 0 ? "&page=" + page : "")
                            + (limit > 0 ? "&limit=" + limit : "")
                            + (!endDate.isEmpty() ? "&to_time=" + endDate : "")
                    , null);
        }
    }

    public void completePickUp(String taskId, final EventListener listener) {
        if (preference.getUserId().isEmpty())
            TLog.log("Please identify this user with driver id to update.");
        else if (TUtils.isNetNotConnected(context))
            TLog.log("Internet is not Connected");
        else {
            TRestCall tRestCall = new TRestCall(context);
            tRestCall.setHttpType(TRestCall.HTTP_TYPE.TYPE_PATCH);
            tRestCall.setCallBackListener(new TRestCall.ResponseListener() {
                @Override
                public void onResponse(String result) {
                    listener.onSuccess("Task Updated");
                }
            });
            FormBody.Builder builder = new FormBody.Builder();
            builder.add("pickup.is_complete", String.valueOf(true));
            tRestCall.requestApi("task?task_id=" + taskId, builder.build());
        }
    }

    public void completeDrop(String taskId, final EventListener listener) {
        if (preference.getUserId().isEmpty())
            TLog.log("Please identify this user with driver id to update.");
        else if (TUtils.isNetNotConnected(context))
            TLog.log("Internet is not Connected");
        else {
            TRestCall tRestCall = new TRestCall(context);
            tRestCall.setHttpType(TRestCall.HTTP_TYPE.TYPE_PATCH);
            tRestCall.setCallBackListener(new TRestCall.ResponseListener() {
                @Override
                public void onResponse(String result) {
                    listener.onSuccess("Task Updated");
                }
            });
            FormBody.Builder builder = new FormBody.Builder();
            builder.add("drop.is_complete", String.valueOf(true));
            tRestCall.requestApi("task?task_id=" + taskId, builder.build());
        }
    }

    public void updateProperty(String key, String value) {
        try {
            if (preference.getUserId().isEmpty())
                TLog.log("Please identify this user with driver id to set him property.");
            else if (TUtils.isNetNotConnected(context))
                TLog.log("Internet is not Connected");
            else {
                TRestCall tRestCall = new TRestCall(context);
                tRestCall.setHttpType(TRestCall.HTTP_TYPE.TYPE_PATCH);
                JSONObject jsonObject = new JSONObject();
                JSONObject property = new JSONObject();
                property.put("key", key);
                property.put("value", value);
                jsonObject.put("property", property);
                tRestCall.requestApi("update_user?driver_id=" + preference.getUserId()
                        , createBody(jsonObject.toString()));
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void updateTaskProperty(String taskID, String key, String value) {
        try {
            TRestCall tRestCall = new TRestCall(context);
            tRestCall.setHttpType(TRestCall.HTTP_TYPE.TYPE_PATCH);
            JSONObject jsonObject = new JSONObject();
            JSONObject property = new JSONObject();
            property.put("key", key);
            property.put("value", value);
            jsonObject.put("property", property);
            tRestCall.requestApi("task?task_id=" + taskID,
                    createBody(jsonObject.toString()));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private static RequestBody createBody(String data) {
        return RequestBody.create(data, MediaType.parse("application/json; charset=utf-8"));
    }

}