package io.teliver.sdk.core;

import android.annotation.SuppressLint;
import android.content.Context;

import com.google.firebase.messaging.RemoteMessage;
import io.teliver.sdk.models.PushData;
import io.teliver.sdk.models.Task;
import io.teliver.sdk.models.TaskOptions;
import io.teliver.sdk.models.TrackingOptions;
import io.teliver.sdk.models.Trip;
import io.teliver.sdk.models.TripBuilder;
import io.teliver.sdk.models.TripOptions;
import io.teliver.sdk.models.User;
import io.teliver.sdk.util.SdkHandler;
import io.teliver.sdk.util.TUtils;
import io.teliver.sdk.util.TeliverMap;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

public class Teliver {

    @SuppressLint("StaticFieldLeak")
    private static SdkHandler handler;

    private Teliver() {
        // Single ton Private Constructor
    }

    public static void init(Context context, String apiKey) {
        if (context == null)
            TLog.log("Context should not be null");
        else if (TUtils.clearNull(apiKey).isEmpty())
            TLog.log("APP Key is Null or Empty");
        else {
            handler = SdkHandler.getInstance(context);
            handler.initializeClient(apiKey);
        }
    }

    public static void setInitStatusListener(InitStatusListener initListener) {
        if (handler == null)
            TLog.log("Teliver Client is not initialized. call init()");
        else handler.setInitListener(initListener);
    }

    public static void identifyUser(User user) {
        if (handler == null)
            TLog.log("Teliver Client is not initialized. call init()");
        else handler.identifyDeviceUser(user);
    }

    public static void unregisterUser() {
        if (handler == null)
            TLog.log("Teliver Client is not initialized. call init()");
        else handler.unRegisterUser();
    }

    public static void startTrip(String trackingID) {
        startTrip(new TripBuilder(trackingID).build());
    }

    public static void startTrip(TripOptions options) {
        if (handler == null)
            TLog.log("Teliver Client is not initialized. call init()");
        else {
            handler.startTripUpdates(options);
            TLog.log("Start trip updates");
        }
    }

    public static void stopTrip(String trackingId) {
        if (handler == null)
            TLog.log("Teliver Client is not initialized. call init()");
        else
            handler.stopTripUpdates(trackingId);
    }

    public static void startTracking(TrackingOptions options) {
        if (handler == null)
            TLog.log("Teliver Client is not initialized. call init()");
        else
            handler.startTracking(options);
    }

    public static void stopTracking(String trackingId) {
        List<String> ids = new ArrayList<>();
        ids.add(TUtils.clearNull(trackingId));
        stopTracking(ids);
    }


    public static void stopTracking(List<String> trackingId) {
        if (handler == null)
            TLog.log("Teliver Client is not initialized. call init()");
        else
            handler.stopTracking(trackingId);
    }

    public static void setTripListener(TripListener listener) {
        if (handler == null)
            TLog.log("Teliver Client is not initialized. call init()");
        else
            handler.setListener(listener);
    }

    public static List<Trip> getCurrentTrips() {
        if (handler == null) {
            TLog.log("Teliver Client is not initialized. call init()");
            return null;
        } else return handler.getCurrentTrips();

    }

    public static void sendEventPush(String trackingId, PushData pushData, String tag) {
        sendEventPush(trackingId, pushData);
        tagLocation(trackingId, tag);
    }

    public static void sendEventPush(String trackingId, PushData pushData) {
        if (handler == null)
            TLog.log("Teliver Client is not initialized. call init()");
        else
            handler.sendEventPush(trackingId, pushData);
    }

    public static void tagLocation(String trackingId, String tag) {
        if (handler == null)
            TLog.log("Teliver Client is not initialized. call init()");
        else
            handler.addTag(trackingId, tag);
    }

    public static boolean isTeliverPush(RemoteMessage message) {
        if (message == null || message.getData() == null)
            return false;
        Map<String, String> data = message.getData();
        String key = "is_from";
        return data.containsKey(key) &&
                "Teliver".equalsIgnoreCase(TUtils.clearNull(data.get(key)));
    }

    public static SdkHandler getHandler() {
        return handler;
    }

    public static Class getMap() {
        return TeliverMap.class;
    }

    public static void closeMap() {
        if (handler == null)
            TLog.log("Teliver Client is not initialized. call init()");
        else handler.closeMap();
    }

    public static void getMyTaskList(String status, int page, int limit, TaskListListener listener) {
        if (handler == null)
            TLog.log("Teliver Client is not initialized. call init()");
        else
            handler.getMyTaskList(status, page, limit, listener);
    }

    public static void getMyTaskWithTaskId(String taskId, TaskListener listener) {
        if (handler == null)
            TLog.log("Teliver Client is not initialized. call init()");
        else
            handler.getMyTaskWithTaskId(taskId, listener);
    }

    public static void startTask(final String taskId, final EventListener listener) {
        if (handler == null)
            TLog.log("Teliver Client is not initialized. call init()");
        else if (TUtils.clearNull(taskId).isEmpty())
            TLog.log("Task id cannot be empty");
        else {
            getMyTaskWithTaskId(taskId, new TaskListener() {
                @Override
                public void onSuccess(Task task) {
                    startTrip(new TripBuilder(taskId).build());
                    handler.updateTaskStatus(taskId, "in_progress", listener);
                }

                @Override
                public void onFailure(String reason) {
                    listener.onFailure(reason);
                }
            });
        }
    }

    public static void completeTask(final String taskId, final EventListener listener) {
        if (handler == null)
            TLog.log("Teliver Client is not initialized. call init()");
        else if (TUtils.clearNull(taskId).isEmpty()) {
            TLog.log("Task id cannot be empty");
            listener.onFailure("Task id cannot be empty");
        } else {
            getMyTaskWithTaskId(taskId, new TaskListener() {
                @Override
                public void onSuccess(Task task) {
                    stopTrip(taskId);
                    handler.updateTaskStatus(taskId, "completed", listener);
                }

                @Override
                public void onFailure(String reason) {
                    listener.onFailure(reason);
                }
            });
        }
    }

    public static void acceptTask(final String taskId, final EventListener listener) {
        if (handler == null)
            TLog.log("Teliver Client is not initialized. call init()");
        else if (TUtils.clearNull(taskId).isEmpty())
            TLog.log("Task id cannot be empty");
        else {
            getMyTaskWithTaskId(taskId, new TaskListener() {
                @Override
                public void onSuccess(Task task) {
                    if (taskId.equalsIgnoreCase(task.getTaskId())) {
                        handler.updateTaskStatus(taskId, "accepted", listener);
                    }
                }

                @Override
                public void onFailure(String reason) {
                    listener.onFailure(reason);
                }
            });
        }
    }


    public static void rejectTask(final String taskId, final EventListener listener) {
        if (handler == null)
            TLog.log("Teliver Client is not initialized. call init()");
        else if (TUtils.clearNull(taskId).isEmpty())
            TLog.log("Task id cannot be empty");
        else {
            getMyTaskWithTaskId(taskId, new TaskListener() {
                @Override
                public void onSuccess(Task task) {
                    if (taskId.equalsIgnoreCase(task.getTaskId())) {
                        handler.updateTaskStatus(taskId, "rejected", listener);
                    }
                }

                @Override
                public void onFailure(String reason) {
                    listener.onFailure(reason);
                }
            });
        }
    }

    public static void updateDriverAvailability(boolean status) {
        updateDAvailability(status, null);
    }

    public static void updateDriverAvailability(boolean status, EventListener listener) {
        updateDAvailability(status, listener);
    }

    private static void updateDAvailability(boolean status, EventListener listener) {
        if (handler == null)
            TLog.log("Teliver Client is not initialized. call init()");
        else
            handler.updateAvailability(status, listener);
    }

    public static void cancelTask(final String taskId, final EventListener listener) {
        if (handler == null)
            TLog.log("Teliver Client is not initialized. call init()");
        else if (TUtils.clearNull(taskId).isEmpty())
            TLog.log("Task id cannot be empty");
        else {
            getMyTaskWithTaskId(taskId, new TaskListener() {
                @Override
                public void onSuccess(Task task) {
                    handler.updateTaskStatus(taskId, "cancelled", listener);
                }

                @Override
                public void onFailure(String reason) {
                    listener.onFailure(reason);
                }
            });
        }
    }

    public static void getMyTask(TaskOptions taskOptions) {
        if (handler == null)
            TLog.log("Teliver Client is not initialized. call init()");
        else if (taskOptions == null)
            TLog.log("TaskOptions is null");
        else
            handler.getMyTask(taskOptions);
    }

    public static void completePickupTask(final String taskId
            , final EventListener listener) {
        if (handler == null)
            TLog.log("Teliver Client is not initialized. call init()");
        else if (TUtils.clearNull(taskId).isEmpty())
            TLog.log("Task id cannot be empty");
        else {
            getMyTaskWithTaskId(taskId, new TaskListener() {
                @Override
                public void onSuccess(Task task) {
                    handler.completePickUp(taskId, listener);
                }

                @Override
                public void onFailure(String reason) {
                    listener.onFailure(reason);
                }
            });
        }
    }

    public static void completeDropTask(final String taskId
            , final EventListener listener) {
        if (handler == null)
            TLog.log("Teliver Client is not initialized. call init()");
        else if (TUtils.clearNull(taskId).isEmpty())
            TLog.log("Task id cannot be empty");
        else {
            getMyTaskWithTaskId(taskId, new TaskListener() {
                @Override
                public void onSuccess(Task task) {
                    handler.completeDrop(taskId, listener);
                }

                @Override
                public void onFailure(String reason) {
                    listener.onFailure(reason);
                }
            });
        }
    }

    public static void setDriverProperty(String key, String value) {
        if (handler == null)
            TLog.log("Teliver Client is not initialized. call init()");
        else handler.updateProperty(key, value);
    }

    public static void setTaskProperty(final String taskID, final String key, final String value) {
        if (handler == null)
            TLog.log("Teliver Client is not initialized. call init()");
        else {
            getMyTaskWithTaskId(taskID, new TaskListener() {
                @Override
                public void onSuccess(Task task) {
                    handler.updateTaskProperty(taskID, key, value);
                }

                @Override
                public void onFailure(String reason) {
                    TLog.log(reason);
                }
            });
        }
    }

}


