package io.teliver.sdk.core;

import android.annotation.SuppressLint;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.app.Service;
import android.content.Context;
import android.content.Intent;
import android.graphics.Color;
import android.location.Location;
import android.os.Build;
import android.os.IBinder;
import android.os.Looper;

import androidx.annotation.Nullable;
import androidx.core.app.NotificationCompat;

import com.google.android.gms.location.FusedLocationProviderClient;
import com.google.android.gms.location.LocationCallback;
import com.google.android.gms.location.LocationListener;
import com.google.android.gms.location.LocationRequest;
import com.google.android.gms.location.LocationResult;
import com.google.android.gms.location.LocationServices;
import com.google.android.gms.tasks.OnSuccessListener;
import com.google.gson.Gson;
import io.teliver.sdk.models.NotificationData;
import io.teliver.sdk.models.PushData;
import io.teliver.sdk.models.TConstants;
import io.teliver.sdk.models.TLocation;
import io.teliver.sdk.models.TMessage;
import io.teliver.sdk.models.TTrip;
import io.teliver.sdk.models.Trip;
import io.teliver.sdk.models.TripHistory;
import io.teliver.sdk.models.TripOptions;
import io.teliver.sdk.util.MessageClient;
import io.teliver.sdk.util.SdkHandler;
import io.teliver.sdk.util.ServiceBinder;
import io.teliver.sdk.util.TConverter;
import io.teliver.sdk.util.TPreference;
import io.teliver.sdk.util.TRestCall;
import io.teliver.sdk.util.TUtils;

import org.eclipse.paho.client.mqttv3.IMqttActionListener;
import org.eclipse.paho.client.mqttv3.IMqttDeliveryToken;
import org.eclipse.paho.client.mqttv3.IMqttToken;
import org.eclipse.paho.client.mqttv3.MqttMessage;
import org.json.JSONArray;
import org.json.JSONObject;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import okhttp3.FormBody;
import okhttp3.RequestBody;

import static io.teliver.sdk.models.TConstants.TYPE_CMD;
import static io.teliver.sdk.models.TConstants.TYPE_LOCATION;
import static io.teliver.sdk.util.TUtils.getGson;
@SuppressLint("MissingPermission")
public class TService extends Service implements LocationListener, IMqttActionListener {

    private FusedLocationProviderClient providerClient;

    private TPreference preference;

    private LocationUpdates locationUpdates;

    private ServiceBinder binder;

    private TripListener tripListener;

    private String agentId;

    private Location lastKnown;

    private MessageClient messageClient;

    private Map<String, Trip> currentTrips = new HashMap<>();

    private Location lastUsed = null, lastSent = null;

    private float lastDifference = 0.0f, lastBearing = 0.0f;

    private ArrayList<Trip> trips = new ArrayList<>();

    private boolean availability = false;

    private long lastUpdate = System.currentTimeMillis();

    @Override
    public void onCreate() {
        super.onCreate();
        SdkHandler handler = Teliver.getHandler();
        preference = new TPreference(getApplicationContext());
        if (handler == null) {
            TLog.log("Teliver Client is not initialized. call init()");
            stopSelf();
        } else {
            binder = new ServiceBinder(this);
            providerClient = LocationServices.getFusedLocationProviderClient(this);
        }
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        messageClient = MessageClient.getInstance(this);
        messageClient.setConnectionListener(this);
        TUtils.connectClient(this, messageClient);
        return START_STICKY;
    }

    @Nullable
    @Override
    public IBinder onBind(Intent intent) {
        return binder;
    }

    public void prepareUpdates(final TripOptions options, final String agentId) {
        try {
            this.agentId = agentId;
            providerClient.getLastLocation().addOnSuccessListener(new OnSuccessListener<Location>() {
                @Override
                public void onSuccess(final Location location) {
                    if (location != null) {
                        TConverter converter = new TConverter(TService.this);
                        converter.convertLatLng(location, new TConverter.Converter() {
                            @Override
                            public void onLocationString(String address) {
                                String startsAt = location.getLatitude() + "," + location.getLongitude();
                                callStartTrip(options, startsAt, address);
                            }
                        });
                    } else callStartTrip(options, "", "");
                }
            });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void callStartTrip(TripOptions options, String startsAt, String address) {
        TRestCall tRestCall = new TRestCall(this);
        tRestCall.setCallBackListener(new TripStartListener(options));
        FormBody.Builder builder = new FormBody.Builder();
        builder.add("agent_id", TUtils.clearNull(agentId));
        builder.add("start_location", address);
        builder.add("starts_at", startsAt);
        builder.add("tracking_id", options.getTrackingId());
        tRestCall.requestApi("trip", builder.build());
    }

    public void stopTripUpdates(final String trackingId) {
        try {
            sendMessage(createPayload(trackingId), trackingId);
            providerClient.getLastLocation().addOnSuccessListener(new OnSuccessListener<Location>() {
                @Override
                public void onSuccess(final Location location) {
                    if (location != null) {
                        TConverter converter = new TConverter(TService.this);
                        converter.convertLatLng(location, new TConverter.Converter() {
                            @Override
                            public void onLocationString(String address) {
                                String endsAt = location.getLatitude() + "," + location.getLongitude();
                                stopUpdates(trackingId, endsAt, address);
                            }
                        });
                    } else stopUpdates(trackingId, "", "");
                }
            });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void stopUpdates(String id, String endsAt, String address) {
        TRestCall tRestCall = new TRestCall(this);
        tRestCall.setHttpType(TRestCall.HTTP_TYPE.TYPE_PATCH);
        FormBody.Builder builder = new FormBody.Builder();
        builder.add("end_location", address);
        builder.add("ends_at", endsAt);
        builder.add("tracking_id", id);
        builder.add("agent_id", TUtils.clearNull(agentId));
        tRestCall.setCallBackListener(new TripStopListener(id));
        tRestCall.requestApi("trip", builder.build());
    }

    @Override
    public void onSuccess(IMqttToken asyncActionToken) {
        TLog.log("Client connection Success:");
    }

    @Override
    public void onFailure(IMqttToken asyncActionToken, Throwable exception) {
        TLog.log("Client Failure:" + exception.getMessage());
    }

    private class TripStartListener implements TRestCall.ResponseListener {

        private TripOptions options;

        private Gson gson;

        TripStartListener(TripOptions options) {
            this.options = options;
            gson = getGson();
        }

        @Override
        public void onResponse(String result) {
            try {
                if (result.isEmpty() && tripListener != null) {
                    tripListener.onTripError("Server Error");
                    return;
                }
                TLog.log(result);
                TTrip response = gson.fromJson(result, TTrip.class);
                if (response.isSuccess()) {
                    currentTrips.put(response.getTripData().getTrackingId(), response.getTripData());
                    if (tripListener != null)
                        tripListener.onTripStarted(response.getTripData());
                    sendTripPush();
                    showForeGroundInfo(options);
                } else if (tripListener != null)
                    tripListener.onTripError(response.getMessage());
            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        private void sendTripPush() {
            try {
                PushData pushData = options.getPushData();
                if (pushData == null || pushData.getUsers().length == 0)
                    return;
                NotificationData data = new NotificationData();
                data.setCommand(TConstants.CMD_TRIP_START);
                data.setMessage(pushData.getMessage());
                data.setPayload(pushData.getPayload());
                data.setTrackingID(options.getTrackingId());
                RequestBody body = new FormBody.Builder()
                        .add(TConstants.PUSH_TITLE, pushData.getMessage())
                        .add(TConstants.PUSH_DATA, gson.toJson(data))
                        .add(TConstants.PUSH_IDS, gson.toJson(pushData.getUsers()))
                        .build();
                TRestCall tRestCall = new TRestCall(TService.this);
                tRestCall.setCallBackListener(new TRestCall.ResponseListener() {
                    @Override
                    public void onResponse(String result) {
                        TLog.log("PUSH RESULT::>>>" + result);
                    }
                });
                tRestCall.requestApi("send_push", body);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    private class TripStopListener implements TRestCall.ResponseListener {

        String trackingId;

        TripStopListener(String trackingId) {
            this.trackingId = trackingId;
        }

        @Override
        public void onResponse(String result) {
            try {
                currentTrips.remove(trackingId);
                if (tripListener != null)
                    tripListener.onTripEnded(trackingId);
                validateAndStopLocationRequest();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    private void showForeGroundInfo(TripOptions tripOptions) {
        try {
            LocationRequest locationRequest = tripOptions.getLocationRequest();
            if (locationRequest == null) {
                locationRequest = LocationRequest.create();
                locationRequest.setInterval(tripOptions.getInterval());
                locationRequest.setSmallestDisplacement(tripOptions.getDistance());
                locationRequest.setPriority(LocationRequest.PRIORITY_HIGH_ACCURACY);
            }
            locationUpdates = new LocationUpdates();
            providerClient.requestLocationUpdates(locationRequest,
                    locationUpdates, Looper.myLooper());

            NotificationCompat.Builder notBuilder = new NotificationCompat
                    .Builder(this, getApplicationContext().getPackageName());
            notBuilder.setContentTitle(TUtils.getApplicationName(getApplicationContext()));
            notBuilder.setContentText(TUtils.getApplicationName(getApplicationContext())
                    + " is utilizing location services.");
            notBuilder.setSmallIcon(getApplicationContext().getResources().
                    getIdentifier("ic_stat_name", "drawable", getApplicationContext()
                            .getPackageName()));

            notBuilder.setAutoCancel(false);
            notBuilder.setPriority(NotificationCompat.PRIORITY_MIN);
            Intent intent = getPackageManager().getLaunchIntentForPackage(getPackageName());
            if (intent != null) {
                intent.addFlags(Intent.FLAG_ACTIVITY_CLEAR_TASK | Intent.FLAG_ACTIVITY_NEW_TASK);
                PendingIntent pendingIntent = PendingIntent.getActivity(this, 0,
                        intent, PendingIntent.FLAG_UPDATE_CURRENT);
                notBuilder.setContentIntent(pendingIntent);
                setNotificationChannel(notBuilder);
                startForeground(909, notBuilder.build());
            } else
                TLog.log("No Launcher intent found to set notification pending intent");
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void setNotificationChannel(NotificationCompat.Builder builder) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            String channelId = "notification_channel_1";
            NotificationChannel channel = new NotificationChannel(channelId
                    , TUtils.getApplicationName(getApplicationContext())
                    , NotificationManager.IMPORTANCE_DEFAULT);
            channel.setShowBadge(true);
            channel.shouldShowLights();
            channel.setLightColor(Color.BLUE);
            channel.canBypassDnd();
            channel.setDescription("");
            builder.setChannelId(channelId);
            builder.setNumber(1);
            NotificationManager manager = (NotificationManager)
                    getSystemService(Context.NOTIFICATION_SERVICE);
            if (manager != null)
                manager.createNotificationChannel(channel);
        }
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        try {
            if (messageClient != null && messageClient.isConnected())
                messageClient.disconnect();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public void onLocationChanged(Location location) {
        if (location == null)
            return;
        if (location.getAccuracy() >= 2 && location.getAccuracy() <= 30) {
            processLocation(location);
            if (tripListener != null)
                tripListener.onLocationUpdate(location);
        }
    }

    public void updateListener(TripListener listener) {
        this.tripListener = listener;
    }

    public void updateAvailability(boolean availability) {
        this.availability = availability;
        if (availability)
            showForeGroundInfo(new TripOptions("monitor"));
        else
            validateAndStopLocationRequest();

    }

    public List<Trip> getCurrentTrips() {
        trips.clear();
        trips.addAll(currentTrips.values());
        return trips;
    }

    private String createLocationPayload(Location location, Trip trip, boolean
            shouldSave, Location saveLocation, String tag) {
        TMessage tMessage = new TMessage();
        tMessage.setType(TYPE_LOCATION);
        tMessage.setTrackingId(trip.getTrackingId());
        TLocation tLocation = new TLocation();
        tLocation.setLatitude(location.getLatitude());
        tLocation.setLongitude(location.getLongitude());
        if (lastKnown != null)
            tLocation.setBearing(lastKnown.bearingTo(location));
        tMessage.setLocation(tLocation);
        tMessage.setShouldSave(shouldSave);
        tMessage.setTripId(trip.getTripId());
        tMessage.setTtl(trip.getTtl());
        if (shouldSave)
            tMessage.setTripHistory(new TripHistory(saveLocation.
                    getLatitude(), saveLocation.getLongitude(), tag));
        return getGson().toJson(tMessage);
    }

    private String createPayload(String trackingId) {
        TMessage message = new TMessage();
        message.setType(TYPE_CMD);
        message.setTrackingId(trackingId);
        return getGson().toJson(message);
    }

    private void sendMessage(String payload, String trackingId) {
        MqttMessage mqttMessage = new MqttMessage();
        mqttMessage.setQos(1);
        mqttMessage.setPayload(payload.getBytes());
        messageClient.publishData(trackingId, mqttMessage, new IMqttActionListener() {
            @Override
            public void onSuccess(IMqttToken asyncActionToken) {
                TLog.log("onSuccess");
            }

            @Override
            public void onFailure(IMqttToken asyncActionToken, Throwable exception) {
                try {
                    if (TUtils.isNetNotConnected(getBaseContext()))
                        return;
                    String msgBody;
                    byte[] content = ((IMqttDeliveryToken) asyncActionToken).getMessage().getPayload();
                    msgBody = new String(content, StandardCharsets.UTF_8);
                    TMessage message = getGson().fromJson(msgBody, TMessage.class);
                    stopTripUpdates(message.getTrackingId());
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        });
    }

    public void tagLocation(final String trackingId, final String tag) {
        final Location[] tagLoc = {lastKnown};
        providerClient.getLastLocation().addOnSuccessListener(new OnSuccessListener<Location>() {
            @Override
            public void onSuccess(Location location) {
                if (location != null && tag == null)
                    tagLoc[0] = location;
                if (currentTrips.containsKey(trackingId) && tagLoc[0] != null) {
                    Trip trip = currentTrips.get(trackingId);
                    if (trip != null)
                        sendMessage(createLocationPayload(tagLoc[0], trip, true, tagLoc[0], tag), trackingId);
                } else
                    TLog.log("Couldn't get Current Location or Trip with the given id");
            }
        });
    }

    private void processLocation(Location currentLocation) {
        float ms = 0.0f;
        if (lastKnown != null) {
            long diffInMs = currentLocation.getTime() - lastKnown.getTime();
            long diffInSec = TimeUnit.MILLISECONDS.toSeconds(diffInMs);
            float distance = lastKnown.distanceTo(currentLocation);
            ms = distance / diffInSec;
        }
        if (ms < 100.0f) {
            boolean shouldSave = false;
            if (lastSent == null) {
                lastSent = currentLocation;
                shouldSave = true;
            } else {
                float currentBearing = lastSent.bearingTo(currentLocation);
                if (Float.compare(0.0f, lastBearing) != 0) {
                    float currentDifference = Math.abs(lastBearing - currentBearing);
                    if (Float.compare(0.0f, lastDifference) != 0) {
                        float difference = Math.abs(lastDifference - currentDifference);
                        int compare = Float.compare(difference, 5.0f);
                        if (compare > 0 && lastUsed != null) {
                            lastSent.set(lastUsed);
                            lastDifference = 0.0f;
                            shouldSave = true;
                        } else
                            lastUsed = currentLocation;
                    } else
                        lastDifference = currentDifference;
                } else
                    lastBearing = currentBearing;
            }

            Set<String> trackingIds = currentTrips.keySet();
            for (String trackingId : trackingIds) {
                Trip trip = currentTrips.get(trackingId);
                if (trip != null)
                    sendMessage(createLocationPayload(currentLocation, trip
                            , shouldSave, lastSent, ""), trackingId);
            }
            if ((availability && shouldSave) || (availability && lastKnown == null) || System.currentTimeMillis() - lastUpdate >= 10000) {
                lastUpdate = System.currentTimeMillis();
                updateDriverLocation(currentLocation);
            }
        }
        lastKnown = currentLocation;
    }

    private void updateDriverLocation(Location currentLocation) {
        try {
            TRestCall tRestCall = new TRestCall(this);
            tRestCall.setHttpType(TRestCall.HTTP_TYPE.TYPE_PATCH);

            FormBody.Builder builder = new FormBody.Builder();
            JSONObject obj = new JSONObject();
            JSONArray location = new JSONArray();
            location.put(currentLocation.getLongitude());
            location.put(currentLocation.getLatitude());
            obj.put("lnglat", location);
            builder.add("location", obj.toString());
            tRestCall.requestApi("update_user?driver_id=" + preference.
                    getUserId(), builder.build());
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void validateAndStopLocationRequest() {
        if (currentTrips.isEmpty() && !availability) {
            providerClient.removeLocationUpdates(locationUpdates);
            stopSelf();
            stopForeground(true);
        }
    }

    private class LocationUpdates extends LocationCallback {
        @Override
        public void onLocationResult(LocationResult locationResult) {
            super.onLocationResult(locationResult);
            Location location = locationResult.getLastLocation();
            if (location != null && isAccuracyOk(location.getAccuracy()))
                processLocation(location);
        }

        private boolean isAccuracyOk(float accuracy) {
            return accuracy >= 2 && accuracy <= 30;
        }
    }
}
