package org.codehaus.mojo.aspectj;

/**
 * The MIT License
 *
 * Copyright 2005-2006 The Codehaus.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of
 * this software and associated documentation files (the "Software"), to deal in
 * the Software without restriction, including without limitation the rights to
 * use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is furnished to do
 * so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.codehaus.plexus.util.DirectoryScanner;
import org.codehaus.plexus.util.Scanner;

/**
 * Weaves all test classes.
 * 
 * AspectJ Compiler Plugin.
 *
 * @author <a href="mailto:kaare.nilsen@gmail.com">Kaare Nilsen</a>
 */
@Mojo( name="test-compile", threadSafe = true, defaultPhase = LifecyclePhase.TEST_COMPILE, requiresDependencyResolution = ResolutionScope.TEST )
public class AjcTestCompileMojo
    extends AbstractAjcCompiler
{
    protected static final String MAVEN_TEST_SKIP = "maven.test.skip";

    /**
     * Flag to indicate if the main source dirs should be a part of the compile process.
     * <strong>Note!</strong> This will make all classes in main source dir appear in the
     * test output dir also, potentially overwriting test resources.
     *
     */
    @Parameter( defaultValue = "false" )
    protected boolean weaveMainSourceFolder = false;

    /**
     * Flag to indicate if aspects in the the main source dirs
     * should be a part of the compile process
     *
     */
    @Parameter( defaultValue = "true" )
    protected boolean weaveWithAspectsInMainSourceFolder = true;
    
    /**
     * The directory where compiled test classes go.
     *
     */
    @Parameter( readonly = true, required = true, defaultValue = "${project.build.testOutputDirectory}" )
    private File outputDirectory;

    /**
     * The directory for sources generated by annotation processing.
     */
    @Parameter( defaultValue = "${project.build.directory}/generated-test-sources/aspectj-maven-plugin" )
    private File generatedTestSourcesDirectory;

    /**
     * <p>
     * Set the java test source folders to use, specifying the includes and excludes.
     * </p> 
     * <p>
     * If you don't specify this parameter, all java test sources of the current project fill be used.
     * If you specify this parameter as an empty tag (i.e. &lt;testSources/&gt;), all test source folders will be ignored.
     * Otherwise specify the test source folder(s) to use.
     * <p>
     * 
     * @since 1.4
     * @see DirectoryScanner
     */
    @Parameter
    private Scanner[] testSources;

    @Override
    public void execute() throws MojoExecutionException
    {
        if (isSkipTestCompile())
        {
            getLog().info("Not compiling test sources");
            return;
        }
        super.execute();
    }

    @Override
    protected List<String> getClasspathDirectories()
    {
        List<String> outputDirectories = new ArrayList<String>();
        outputDirectories.add( project.getBuild().getTestOutputDirectory() );
        outputDirectories.add( project.getBuild().getOutputDirectory() );
        return outputDirectories;
    }
    
    @Override
    protected File getOutputDirectory()
    {
        return outputDirectory;
    }

    @Override
    public File getGeneratedSourcesDirectory() {
        return generatedTestSourcesDirectory;
    }

    @SuppressWarnings( "unchecked" )
    @Override
    protected List<String> getSourceDirectories()
    {
        List<String> sourceDirs = new ArrayList<String>();
        sourceDirs.addAll( project.getTestCompileSourceRoots() );
        if ( weaveMainSourceFolder )
        {
            sourceDirs.addAll( project.getCompileSourceRoots() );
        }
        return sourceDirs;
    }

    protected Scanner[] getJavaSources()
    {
        return testSources;
    }

    protected String getAdditionalAspectPaths()
    {
        String additionalPath = null;
        if ( weaveWithAspectsInMainSourceFolder )
        {
            additionalPath = project.getBuild().getOutputDirectory();
        }
        return additionalPath;
    }

    private boolean isSkipTestCompile()
    {
        String skipTestCompile = System.getProperty(MAVEN_TEST_SKIP);
        return Boolean.parseBoolean(skipTestCompile);
    }
}
