package io.solidtech.crash.video;

import android.content.res.AssetManager;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.os.Handler;
import android.util.Log;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;

import io.solidtech.crash.SolidClient;
import io.solidtech.crash.SolidConstants;
import io.solidtech.crash.SolidThread;
import io.solidtech.crash.entities.sugar.VideoMaterial;
import io.solidtech.crash.utils.DebugUtils;

/**
 * Created by vulpes on 16. 1. 19..
 */
public class VideoGenerator {

    public interface OnVideoGeneratedListener {
        void onVideoGenerated(long videoMaterialId, File file);
    }

    private static final String TAG = VideoGenerator.class.getSimpleName();
    private static final boolean VERBOSE = true;
    private static final DebugUtils.DebugLogger sLogger = new DebugUtils.DebugLogger(TAG, VERBOSE);

    private final SolidClient mClient;
    private final OnVideoGeneratedListener mListener;
    private final AtomicBoolean mEnabled = new AtomicBoolean(false);
    private VideoGeneratorThread mThread;
    private ConcurrentQueue<VideoGenerateTask> mQueue = new ConcurrentQueue<>();
    private Set<Long> mPushedMaterialIds = Collections.synchronizedSet(new HashSet<Long>());

    public VideoGenerator(SolidClient client, OnVideoGeneratedListener listener) {
        mListener = listener;
        mClient = client;
    }

    public synchronized void start() {
        if (mThread != null && mThread.isAlive()) {
            return;
        }
        mThread = new VideoGeneratorThread();
        mThread.setDaemon(true);
        mThread.start();
    }

    public synchronized void stop() {
        if (mThread != null && mThread.isAlive()) {
            mThread.interrupt();
            mThread = null;
        }
    }

    public void pushVideoMaterial(long materialId, String outputName) {
        sLogger.d("pushed:" + materialId);

        VideoGenerateTask task = new VideoGenerateTask();
        task.materialId = materialId;
        task.videoFileName = outputName;
        mQueue.push(task);
        mPushedMaterialIds.add(materialId);
    }

    public boolean hasVideoMaterial(long materialId) {
        return mPushedMaterialIds.contains(materialId);
    }

    public void setEnable() {
        mEnabled.set(true);
        synchronized (mEnabled) {
            mEnabled.notifyAll();
        }
    }

    public void setDisable() {
        mEnabled.set(false);
        synchronized (mEnabled) {
            mEnabled.notifyAll();
        }
    }

    private class VideoGeneratorThread extends SolidThread {

        private Handler mHandler;

        public VideoGeneratorThread() {
            super("SolidVideoGeneratorThread");
            mHandler = new Handler();
        }

        @Override
        public void run() {
            super.run();
            try {
                while (true) {
                    sLogger.d("waiting q item..");
                    final VideoGenerateTask task = mQueue.pop();

                    sLogger.d("waiting enable..");
                    while (!mEnabled.get()) {
                        synchronized (mEnabled) {
                            mEnabled.wait();
                        }
                    }

                    File parentDir = VideoMaterial.getExternalStorage(mClient.getApplication(), task.materialId);
                    if (parentDir != null && parentDir.exists()) {
                        sLogger.d("start create video:" + task.materialId);

                        final File file = createVideo(
                                mClient,
                                parentDir,
                                task.videoFileName);

                        if (file != null && file.exists()) {
                            sLogger.d("video created :" + task.materialId);
                            mHandler.post(new Runnable() {
                                @Override
                                public void run() {
                                    // set video file
                                    if (mListener != null) {
                                        mListener.onVideoGenerated(task.materialId, file);
                                    }
                                }
                            });
                        } else {
                            sLogger.d("failed to create video. no material directory exists.");
                        }
                    }
                    mPushedMaterialIds.remove(task.materialId);
                }
            } catch (InterruptedException e) {
                // do nothing
            }
        }
    }

    public static File createVideo(SolidClient client, File dir, String outputName) {
        File[] files = dir.listFiles(new FilenameFilter() {
            @Override
            public boolean accept(File dir, String filename) {
                String ext = SolidConstants.CAPTURE_PHOTO_TYPE == SolidConstants.ImageType.JPEG ? ".jpg" : ".png";
                return filename.endsWith(ext);
            }
        });

        if (files == null) {
            return null;
        }

        Arrays.sort(files, new Comparator<File>() {
            @Override
            public int compare(File lhs, File rhs) {
                return lhs.getName().compareTo(rhs.getName());
            }
        });

        List<File> images = new ArrayList<>(Arrays.asList(files));

        try {
            if (images.size() == 0) {
                AssetManager assetManager = client.getApplication().getAssets();
                InputStream in = assetManager.open("no_video.png");
                File tempFile = File.createTempFile("video_temp", ".png", dir);
                FileOutputStream stream = new FileOutputStream(tempFile);
                byte[] buffer = new byte[1024];
                int read;
                while ((read = in.read(buffer)) != -1) {
                    stream.write(buffer, 0, read);
                }
                in.close();
                stream.close();
                images.add(tempFile);
            }

            File tempFile = File.createTempFile("temp_video_", "gif", dir);
            File videoFile = createVideoAsGif(client, images, tempFile);
            if (videoFile != null) {
                for (File file : images) {
                    file.delete();
                }
            }
            File destFile = new File(dir, outputName);
            if (destFile.exists()) {
                destFile.delete();
            }
            videoFile.renameTo(destFile);
            return destFile;
        } catch (IOException e) {
            return null;
        }
    }

    public static File createVideoAsGif(SolidClient client, List<File> files, File dest) {
        try {

            GifEncoder encoder = new GifEncoder();
            encoder.setFrameRate(client.getConfiguration().videoFrame);
            ByteArrayOutputStream bos = new ByteArrayOutputStream();

            encoder.start(bos);
            int percent = -1;
            for (int i = 0; i < files.size(); i++) {
                File file = files.get(i);
                BitmapFactory.Options options = new BitmapFactory.Options();
                options.inPreferredConfig = Bitmap.Config.RGB_565;
                Bitmap bitmap = BitmapFactory.decodeFile(file.getAbsolutePath(), options);
                if (bitmap != null) {
                    encoder.addFrame(bitmap);
                    bitmap.recycle();
                }
                int newPercent = i * 100 / files.size();
                if (newPercent != percent) {
                    percent = newPercent;
                    sLogger.d("progress:" + percent + "%");
                }
            }

            sLogger.d("create finishing..");
            encoder.finish();

            OutputStream outputStream = new FileOutputStream(dest.getAbsoluteFile());
            outputStream.write(bos.toByteArray());
            outputStream.close();

            return dest;
        } catch (IOException e) {
            e.printStackTrace();
        }

        return null;
    }

    private static class VideoGenerateTask {
        long materialId;
        String videoFileName;
    }
}
