package io.solidtech.crash.utils;

import android.content.Context;
import android.os.Environment;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.nio.channels.FileChannel;
import java.text.DecimalFormat;

import io.solidtech.crash.SolidConstants;

public class FileUtils {

    private static File sSolidDir;

    public static void copy(File src, File dst) throws IOException {
        FileInputStream inStream = new FileInputStream(src);
        FileOutputStream outStream = new FileOutputStream(dst);

        FileChannel inChannel = inStream.getChannel();
        FileChannel outChannel = outStream.getChannel();

        inChannel.transferTo(0, inChannel.size(), outChannel);
        inStream.close();
        outStream.close();
    }

    public static void move(File src, File dst) throws IOException {
        src.renameTo(dst);
    }

    public static void deleteAll(File src) {
        if (src.isDirectory()) {
            for (File child : src.listFiles()) {
                deleteAll(child);
            }
        }
        src.delete();
    }

    public static boolean isSdcardReadable() {
        String state = Environment.getExternalStorageState();
        return Environment.MEDIA_MOUNTED.equals(state) || Environment.MEDIA_MOUNTED_READ_ONLY.equals(state);
    }

    public static boolean isSdcardWritable() {
        return Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED);
    }

    public static String readableFileSize(long size) {
        if (size <= 0) {
            return "0";
        }
        final String[] units = new String[]{"B", "KB", "MB", "GB", "TB"};
        int digitGroups = (int) (Math.log10(size) / Math.log10(1024));
        return new DecimalFormat("#,##0.#").format(size / Math.pow(1024, digitGroups)) + " " + units[digitGroups];
    }

    public static synchronized File getSolidDir(Context context, boolean writable) {
        if (writable && !isSdcardWritable()) {
            return null;
        }
        if (!writable && !isSdcardReadable()) {
            return null;
        }
        if (sSolidDir != null && sSolidDir.exists()) {
            return sSolidDir;
        }
        File externalDir = context.getExternalFilesDir(null);
        if (externalDir == null) {
            return null;
        }
        File solidDir = new File(externalDir, "io.solidtech.crash");
        if (!solidDir.exists()) {
            if (!isSdcardWritable()) {
                return null;
            }
            solidDir.mkdir();
        }
        sSolidDir = solidDir;
        return solidDir;
    }

    public static File getVideoMaterialDir(Context context, boolean writable) {
        File solidDir = getSolidDir(context, writable);
        if (solidDir == null) {
            return null;
        }

        File videoDir = new File(solidDir, SolidConstants.VIDEO_MATERIAL_DIR_NAME);
        if (!videoDir.exists()) {
            if (!isSdcardWritable()) {
                return null;
            }
            videoDir.mkdir();
        }

        return videoDir;
    }

    public static File getCaptureDir(Context context, boolean writable) {
        File solidDir = getSolidDir(context, writable);
        if (solidDir == null) {
            return null;
        }

        File dir = new File(solidDir, SolidConstants.CAPTURED_PHOTO_DIR_NAME);
        if (!dir.exists()) {
            if (!isSdcardWritable()) {
                return null;
            }
            dir.mkdir();
        }

        return dir;
    }

    public static boolean writeStringToFile(File file, String value) throws IOException {
        if (!isSdcardWritable()) {
            throw new IOException("Sdcard not writable");
        }
        if (file.exists()) {
            file.delete();
        }
        FileOutputStream fos = new FileOutputStream(file);
        BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(fos));
        writer.write(value);
        writer.flush();
        writer.close();
        fos.close();
        return true;
    }

    public static String readStringFromFile(File file) throws IOException {
        if (!isSdcardReadable()) {
            throw new IOException("Sdcard not readable");
        }
        if (!file.exists()) {
            return null;
        }
        FileInputStream fis = new FileInputStream(file);
        BufferedReader reader = new BufferedReader(new InputStreamReader(fis));
        String value = reader.readLine();
        fis.close();
        return value;
    }

}
