package io.solidtech.crash.network;

import com.squareup.okhttp.Call;
import com.squareup.okhttp.Request;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import io.solidtech.crash.SolidClient;
import io.solidtech.crash.api.common.Requests;
import io.solidtech.crash.entities.sugar.TrackingInfo;
import io.solidtech.crash.entities.sugar.VideoMaterial;
import io.solidtech.crash.environments.ConnectivityChangeNotifier;
import io.solidtech.crash.utils.DebugUtils;
import io.solidtech.crash.utils.FileUtils;

/**
 * Created by vulpes on 16. 2. 3..
 */
public class TrackingInfoSender {

    private static final String TAG = TrackingInfoSender.class.getSimpleName();
    private static final boolean VERBOSE = true;
    private static final DebugUtils.DebugLogger sLogger = new DebugUtils.DebugLogger(TAG, VERBOSE);

    public interface Observer {
        void onSendSuccess(TrackingInfo info, JSONObject response);

        void onSendFailure(TrackingInfo info, Request request, IOException e);
    }

    public interface OnCrashSentListener {
        void onCrashSent(JSONObject response);
    }

    private static TrackingInfoSender sInstance;

    public static synchronized TrackingInfoSender getInstance(SolidClient client) {
        if (sInstance == null) {
            sInstance = new TrackingInfoSender(client);
        }
        return sInstance;
    }

    private final SolidClient mClient;
    private final Set<Observer> mObservers;
    private final Set<OnCrashSentListener> mOnCrashSentListeners;
    private final Set<Long> mOnTheFlyTrackingInfoIds;
    private final ConnectivityChangeNotifier mConnectivityNotifier;

    private TrackingInfoSender(SolidClient client) {
        mClient = client;

        mObservers = Collections.synchronizedSet(new HashSet<Observer>());
        mOnCrashSentListeners = Collections.synchronizedSet(new HashSet<OnCrashSentListener>());
        mOnTheFlyTrackingInfoIds = new HashSet<>();

        mConnectivityNotifier = ConnectivityChangeNotifier.getInstance(client.getApplication());

        // Set default debug log observer;
        if (VERBOSE) {
            mObservers.add(new Observer() {
                @Override
                public void onSendSuccess(TrackingInfo info, JSONObject response) {
                    sLogger.d("Tracking info upload success id:" + info.getId() +
                            ", type:" + TrackingInfo.getReadableType(info.getType()));
                }

                @Override
                public void onSendFailure(TrackingInfo info, Request request, IOException e) {
                    sLogger.d("Tracking info upload failed id:" + info.getId() +
                            ", type:" + TrackingInfo.getReadableType(info.getType()));
                }
            });
        }
    }

    public void addObserver(Observer observer) {
        mObservers.add(observer);
    }

    public void removeObserver(Observer observer) {
        mObservers.remove(observer);
    }

    private void notifyObserverSuccess(TrackingInfo info, JSONObject resp) {
        for (Observer observer : new HashSet<>(mObservers)) {
            observer.onSendSuccess(info, resp);
        }
    }

    private void notifyObserverFailure(TrackingInfo info, Request request, IOException e) {
        for (Observer observer : new HashSet<>(mObservers)) {
            observer.onSendFailure(info, request, e);
        }
    }

    public void addOnCrashSentListener(OnCrashSentListener listener) {
        mOnCrashSentListeners.add(listener);
    }

    public void removeOnCrashSentListener(OnCrashSentListener listener) {
        mOnCrashSentListeners.remove(listener);
    }

    private void notifyOnCrashSent(JSONObject jsonResp) {
        for (OnCrashSentListener listener : new HashSet<>(mOnCrashSentListeners)) {
            listener.onCrashSent(jsonResp);
        }
    }

    public synchronized void flush() {
        sLogger.d("try flush");

        Long[] idArray = mOnTheFlyTrackingInfoIds.toArray(new Long[mOnTheFlyTrackingInfoIds.size()]);
        List<TrackingInfo> infoList = TrackingInfo.listAllExceptIds(idArray);

        List<TrackingInfo> crashes = new ArrayList<>();
        List<TrackingInfo> httpPackets = new ArrayList<>();

        for (TrackingInfo info : infoList) {
            switch (info.getType()) {
                case TrackingInfo.TYPE_CRASH:
                    crashes.add(info);
                    break;
                case TrackingInfo.TYPE_VIDEO:
                    if (!mClient.getConfiguration().uploadVideoOnlyWifi ||
                            mConnectivityNotifier.isWifiConnected()) {
                        uploadVideo(info);
                    }
                    break;
                case TrackingInfo.TYPE_HTTP_PACKET:
                    httpPackets.add(info);
                    break;
            }
        }
        if (crashes.size() > 0) {
            uploadCrashes(crashes);
        }
        if (httpPackets.size() > 0) {
            uploadHttpPackets(httpPackets);
        }
        sLogger.d("try upload finished");
    }

    private synchronized void uploadCrashes(final List<TrackingInfo> infoList) {
        Call call = null;

        try {
            JSONArray array = new JSONArray();
            for (TrackingInfo info : infoList) {
                array.put(info.getData());
            }
            if (array.length() > 1) {
                call = Requests.buildBulkCrashCall(mClient.getApiKey(), array);
            } else if (array.length() == 1) {
                call = Requests.buildSingleCrashCall(mClient.getApiKey(), array.getJSONObject(0));
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }

        if (call == null) {
            Set<Long> ids = new HashSet<>();
            for (TrackingInfo info : infoList) {
                ids.add(info.getId());
            }
            TrackingInfo.deleteWithIds(ids.toArray(new Long[ids.size()]));
            return;
        }

        TrackingInfo[] infoArray = infoList.toArray(new TrackingInfo[infoList.size()]);
        TrackingInfoCallback callback = new TrackingInfoCallback(infoArray) {
            @Override
            public void onResponse(JSONObject jsonResp) {
                super.onResponse(jsonResp);

                notifyOnCrashSent(jsonResp);
            }
        };

        call.enqueue(callback);
    }

    private synchronized void uploadVideo(final TrackingInfo info) {
        final VideoMaterial.Cache cache = VideoMaterial.Cache.fromJson(info.getData());
        File videoFile = null;

        if (cache != null) {
            videoFile = VideoMaterial.getVideoFile(mClient.getApplication(),
                    cache.getId(),
                    cache.getVideoFileName());
        }

        if (cache == null || cache.isExpired() || videoFile == null || !videoFile.exists()) {
            info.delete();
            if (cache != null) {
                File dir = VideoMaterial.getExternalStorage(mClient.getApplication(), cache.getId());
                if (dir != null && dir.exists()) {
                    FileUtils.deleteAll(dir);
                }
            }
            return;
        }

        TrackingInfoCallback callback = new TrackingInfoCallback(info) {
            @Override
            public void onResponse(JSONObject jsonResp) {
                super.onResponse(jsonResp);

                // remove external files
                File dir = VideoMaterial.getExternalStorage(mClient.getApplication(), cache.getId());
                if (dir != null && dir.exists()) {
                    FileUtils.deleteAll(dir);
                }
            }
        };

        Call call = Requests.buildUploadVideoCall(mClient.getApiKey(),
                cache.getUserId(),
                cache.getCreatedAt(),
                videoFile,
                cache.getCrashId());
        call.enqueue(callback);
    }

    private synchronized void uploadHttpPackets(final List<TrackingInfo> infoList) {

        Call call = null;
        try {
            JSONArray packetJson = new JSONArray();
            for (TrackingInfo info : infoList) {
                packetJson.put(info.getData());
            }

            call = Requests.buildHttpPacketCall(mClient.getApiKey(), packetJson);

        } catch (JSONException e) {
            e.printStackTrace();
        }

        if (call == null) {
            Set<Long> ids = new HashSet<>();
            for (TrackingInfo info : infoList) {
                ids.add(info.getId());
            }
            TrackingInfo.deleteWithIds(ids.toArray(new Long[ids.size()]));
        }

        TrackingInfo[] infoArray = infoList.toArray(new TrackingInfo[infoList.size()]);
        TrackingInfoCallback callback = new TrackingInfoCallback(infoArray);

        call.enqueue(callback);
    }

    private class TrackingInfoCallback extends JsonCallback {

        private final List<TrackingInfo> mInfoList;
        private final Set<Long> mIds = new HashSet<>();

        public TrackingInfoCallback(TrackingInfo... infoArray) {
            mInfoList = Arrays.asList(infoArray);

            for (TrackingInfo info : infoArray) {
                mIds.add(info.getId());

                sLogger.d("Tracking info try upload id:" + info.getId() +
                        ", type:" + TrackingInfo.getReadableType(info.getType()));
            }

            synchronized (TrackingInfoSender.this) {
                mOnTheFlyTrackingInfoIds.addAll(mIds);
            }
        }

        @Override
        public void onResponse(JSONObject jsonResp) {
            // delete all tracking infos from db
            TrackingInfo.deleteWithIds(mIds.toArray(new Long[mIds.size()]));

            // pop from ontheflyset
            synchronized (TrackingInfoSender.this) {
                mOnTheFlyTrackingInfoIds.removeAll(mIds);
            }

            for (TrackingInfo info : mInfoList) {
                notifyObserverSuccess(info, jsonResp);
            }
        }

        @Override
        public void onFailure(Request request, IOException e) {
            e.printStackTrace();

            // pop from ontheflyset
            synchronized (TrackingInfoSender.this) {
                mOnTheFlyTrackingInfoIds.removeAll(mIds);
            }

            for (TrackingInfo info : mInfoList) {
                notifyObserverFailure(info, request, e);
            }
        }
    }
}
