package io.solidtech.crash.environments;

import android.app.ActivityManager;
import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Point;
import android.os.Build;
import android.text.TextUtils;
import android.util.Pair;
import android.view.Display;
import android.view.WindowManager;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.lang.reflect.Field;
import java.util.Locale;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import io.solidtech.crash.SolidClient;

/**
 * Created by vulpes on 2015. 12. 1..
 */
public class DeviceInfo {

    public static final String TAG = DeviceInfo.class.getSimpleName();
    public static int sProcessorCount = -1;

    public static DeviceInfo create(SolidClient client) {
        return new DeviceInfo(client);
    }

    public static final String UNKNOWN = "unknown";
    public final String deviceName;
    public final String deviceModel;
    public final String deviceSerial;
    public final String deviceBrand;

    public final int processorCount;

    public final long totalRamSize;
    public final long freeRamSize;

    public final String screenOrientation;
    public final int screenWidth;
    public final int screenHeight;

    public final String osVersion;
    public final String osVersionName;
    public final int sdk;

    public final String locale;
    public final String readbleLocale;

    private DeviceInfo(SolidClient client) {
        deviceName = Build.DEVICE;
        deviceModel = Build.MODEL;
        deviceSerial = Build.SERIAL;
        String deviceBrand = Build.BRAND;
        if (TextUtils.isEmpty(deviceBrand)) {
            deviceBrand = Build.MANUFACTURER;
        }
        this.deviceBrand = deviceBrand;

        if (sProcessorCount == -1) {
            sProcessorCount = getNumberOfCores();
        }
        processorCount = sProcessorCount;

        Pair<Long, Long> ramInfo = getRamInfo(client.getApplication());
        freeRamSize = ramInfo.first;
        totalRamSize = ramInfo.second;

        screenOrientation = getScreenOrientation(client.getApplication());

        WindowManager wm = (WindowManager) client.getApplication().getSystemService(Context.WINDOW_SERVICE);
        Display display = wm.getDefaultDisplay();
        Point windowSize = new Point();
        display.getSize(windowSize);
        screenWidth = windowSize.x;
        screenHeight = windowSize.y;

        sdk = Build.VERSION.SDK_INT;
        osVersion = Build.VERSION.RELEASE;
        osVersionName = getOsVersionName();

        Locale locale = client.getApplication().getResources().getConfiguration().locale;
        this.locale = locale.toString();
        readbleLocale = locale.getDisplayName(Locale.US);
    }

    public JSONObject toJSONObject() {
        try {
            JSONObject obj = new JSONObject().put("device_name", deviceName)
                    .put("device_model", deviceModel)
                    .put("device_serial", deviceSerial)
                    .put("device_brand", deviceBrand)
                    .put("processor_count", processorCount)
                    .put("total_ram_size", totalRamSize)
                    .put("free_ram_size", freeRamSize)
                    .put("screen_orientation", screenOrientation)
                    .put("screen_width", screenWidth)
                    .put("screen_height", screenHeight)
                    .put("os_version", osVersion)
                    .put("os_version_name", osVersionName)
                    .put("sdk", sdk)
                    .put("locale", locale)
                    .put("readable_locale", readbleLocale);

            return obj;
        } catch (JSONException e) {
            return null;
        }
    }

    private String getScreenOrientation(Context context) {
        int ori = context.getResources().getConfiguration().orientation;
        switch (ori) {
            case Configuration.ORIENTATION_LANDSCAPE:
                return "landscape";
            case Configuration.ORIENTATION_PORTRAIT:
                return "portrait";
        }
        return UNKNOWN;
    }

    private String getOsVersionName() {
        String osVersionName = null;
        Field[] fields = Build.VERSION_CODES.class.getFields();
        for (Field field : fields) {
            String fieldName = field.getName();
            int fieldValue = -1;

            try {
                fieldValue = field.getInt(new Object());
            } catch (IllegalArgumentException | IllegalAccessException | NullPointerException e) {
//                Log.w(TAG, "Failed to retrive os version name");
            }

            if (fieldValue == Build.VERSION.SDK_INT) {
                osVersionName = fieldName;
                break;
            }
        }
        if (TextUtils.isEmpty(osVersionName)) {
            osVersionName = UNKNOWN;
        }
        return osVersionName;
    }

    private int getNumberOfCores() {
        if (Build.VERSION.SDK_INT >= 17) {
            return Runtime.getRuntime().availableProcessors();
        } else {
            // Use saurabh64's answer
            return getNumCoresOldPhones();
        }
    }

    /**
     * Gets the number of cores available in this device, across all processors.
     * Requires: Ability to peruse the filesystem at "/sys/devices/system/cpu"
     *
     * @return The number of cores, or 1 if failed to get result
     */
    private int getNumCoresOldPhones() {
        //Private Class to display only CPU devices in the directory listing
        class CpuFilter implements FileFilter {
            @Override
            public boolean accept(File pathname) {
                //Check if filename is "cpu", followed by a single digit number
                if (Pattern.matches("cpu[0-9]+", pathname.getName())) {
                    return true;
                }
                return false;
            }
        }

        try {
            //Get directory containing CPU info
            File dir = new File("/sys/devices/system/cpu/");
            //Filter to only list the devices we care about
            File[] files = dir.listFiles(new CpuFilter());
            //Return the number of cores (virtual CPU devices)
            return files.length;
        } catch (Exception e) {
            //Default to return 1 core
            return 1;
        }
    }


    public Pair<Long, Long> getRamInfo(Context context) {
        long totalRAM;
        long freeRAM;

        if (Build.VERSION.SDK_INT >= 16) {
            ActivityManager actManager = (ActivityManager) context.getSystemService(Context.ACTIVITY_SERVICE);
            ActivityManager.MemoryInfo memInfo = new ActivityManager.MemoryInfo();
            actManager.getMemoryInfo(memInfo);
            totalRAM = memInfo.totalMem;
            freeRAM = memInfo.availMem;
        } else {

            RandomAccessFile reader;
            try {
                reader = new RandomAccessFile("/proc/meminfo", "r");

                Pattern p = Pattern.compile("(\\d+)");
                Matcher m;
                String load;

                load = reader.readLine().toLowerCase();
                m = p.matcher(load);
                String totalRamVal = "";
                while (m.find()) {
                    totalRamVal = m.group(1);
                }
                totalRAM = (long) Double.parseDouble(totalRamVal);

                if (load.contains("kb")) {
                    totalRAM *= 1024;
                }
                if (load.contains("mb")) {
                    totalRAM *= 1024 * 1024;
                }
                if (load.contains("gb")) {
                    totalRAM *= 1024 * 1024 * 1024;
                }

                load = reader.readLine().toLowerCase();
                m = p.matcher(load);
                String freeRamVal = "";
                while (m.find()) {
                    freeRamVal = m.group(1);
                }
                reader.close();

                freeRAM = (long) Double.parseDouble(freeRamVal);
                if (load.contains("kb")) {
                    freeRAM *= 1024;
                }
                if (load.contains("mb")) {
                    freeRAM *= 1024 * 1024;
                }
                if (load.contains("gb")) {
                    freeRAM *= 1024 * 1024 * 1024;
                }

            } catch (IOException ex) {
                return new Pair<>((long) 0, (long) 0);
            }
        }

        return new Pair<>(freeRAM, totalRAM);
    }
}
