package io.solidtech.crash.entities.sugar;

import android.content.Context;
import android.text.TextUtils;

import com.orm.SugarRecord;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import io.solidtech.crash.SolidConstants;
import io.solidtech.crash.utils.DebugUtils;
import io.solidtech.crash.utils.FileUtils;

/**
 * Created by vulpes on 16. 1. 27..
 */
public class VideoMaterial extends SugarRecord {

    private static final boolean VERBOSE = true;
    private static final String TAG = VideoMaterial.class.getSimpleName();
    private static final DebugUtils.DebugLogger sLogger = new DebugUtils.DebugLogger(TAG, VERBOSE);

    public static VideoMaterial create(Context context,
                                       String userId,
                                       String createdAt,
                                       String crashUUID,
                                       String videoFileName) throws IOException {

        File videoDir = FileUtils.getVideoMaterialDir(context, true);
        if (videoDir == null) {
            throw new IOException("Failed to ensure video directory");
        }

        VideoMaterial material = null;

        try {
            if (TextUtils.isEmpty(createdAt)) {
                Calendar calendar = GregorianCalendar.getInstance();
                SimpleDateFormat format = new SimpleDateFormat(SolidConstants.TIME_FORMAT);
                createdAt = format.format(calendar.getTime());
            }
            material = new VideoMaterial(userId, createdAt, crashUUID, videoFileName);
            material.save();

            File dir = new File(videoDir, String.valueOf(material.getId()));
            dir.mkdir();
        } catch (Throwable t) {
            if (material != null && material.getId() != null) {
                VideoMaterial.deleteExternalStorage(context, material.getId());
                material.delete();
            }
            throw t;
        }

        return material;
    }

    public static boolean deleteExternalStorage(Context context, long materialId) {
        File videoDir = FileUtils.getVideoMaterialDir(context, true);
        if (videoDir == null) {
            return false;
        }

        File dir = new File(videoDir, String.valueOf(materialId));
        if (dir.exists()) {
            FileUtils.deleteAll(dir);
            return true;
        }
        return false;
    }

    public static File getExternalStorage(Context context, long materialId) {
        File videoDir = FileUtils.getVideoMaterialDir(context, true);
        if (videoDir == null) {
            return null;
        }
        return new File(videoDir, String.valueOf(materialId));
    }

    public static File getVideoFile(Context context, long materialId, String videoFileName) {
        File parentDir = getExternalStorage(context, materialId);
        if (parentDir == null || !parentDir.exists()) {
            return null;
        }
        return new File(parentDir, videoFileName);
    }

    public static boolean hasVideoFile(Context context, long materialId, String videoFileName) {
        File video = getVideoFile(context, materialId, videoFileName);
        return video != null && video.exists();
    }

    public static List<VideoMaterial> listLive(Context context) {
        clearExpired(context);
        return VideoMaterial.listAll(VideoMaterial.class);
    }

    public static List<VideoMaterial> clearExpired(Context context) {
        List<VideoMaterial> expired = new ArrayList<>();

        synchronized (VideoMaterial.class) {
            List<VideoMaterial> materials;
            materials = VideoMaterial.listAll(VideoMaterial.class);

            List<String> expiredIds = new ArrayList<>();
            for (VideoMaterial material : materials) {
                sLogger.d("VideoMaterial " + material.getId() + " is fetched");

                File materialDir = getExternalStorage(context, material.getId());
                if (materialDir != null && materialDir.exists() &&
                        materialDir.listFiles().length > 0 && !material.isExpired()) {
                    sLogger.d("VideoMaterial " + material.getId() + " is alive");
                    continue;
                }

                sLogger.d("VideoMaterial " + material.getId() + " should be removed");
                expired.add(material);
                expiredIds.add(String.valueOf(material.getId()));

            }

            String[] ids = expiredIds.toArray(new String[expiredIds.size()]);
            VideoMaterial.deleteAll(
                    VideoMaterial.class,
                    "ID in (" + TextUtils.join(",", Collections.nCopies(ids.length, "?")) + ")",
                    ids);
        }

        for (VideoMaterial material : expired) {
            File materialDir = getExternalStorage(context, material.getId());

            // remove sdcard dir
            if (materialDir != null && materialDir.exists()) {
                FileUtils.deleteAll(materialDir);
            }
        }

        return expired;
    }

    public static Map<String, VideoMaterial> findByCrashUUID(String... uuids) {
        String query = "CRASH_UUID in (" + TextUtils.join(",", Collections.nCopies(uuids.length, "?")) + ")";
        List<VideoMaterial> list = VideoMaterial.find(VideoMaterial.class, query, uuids);

        Map<String, VideoMaterial> map = new HashMap<>();
        for (VideoMaterial material : list) {
            map.put(material.getCrashUUID(), material);
        }
        return map;
    }


    private String videoFileName;
    private String crashUUID;
    private String crashId;
    private String userId;
    private String createdAt;

    public VideoMaterial() {
    }

    private VideoMaterial(String userId, String createdAt, String crashUUID, String videoFileName) {
        this.userId = userId;
        this.createdAt = createdAt;
        this.crashUUID = crashUUID;
        this.videoFileName = videoFileName;
    }

    public boolean isExpired() {
        Date createdAt = getCreatedAt();
        if (createdAt == null) {
            return true;
        }
        Calendar calendar = GregorianCalendar.getInstance();
        Date now = calendar.getTime();
        Date expireTime = new Date(createdAt.getTime() + SolidConstants.VIDEO_EXPIRE_TIME);
        return expireTime.before(now);
    }

    public boolean isReadyToUpload(Context context) {
        return hasVideoFile(context, getId(), videoFileName) &&
                (TextUtils.isEmpty(crashUUID) || !TextUtils.isEmpty(crashId));
    }

    public String getUserId() {
        return userId;
    }

    public String getCrashId() {
        return crashId;
    }

    public void setCrashId(String crashId) {
        this.crashId = crashId;
    }

    public String getCrashUUID() {
        return crashUUID;
    }

    public Date getCreatedAt() {
        try {
            SimpleDateFormat format = new SimpleDateFormat(SolidConstants.TIME_FORMAT);
            return format.parse(createdAt);
        } catch (ParseException e) {
            e.printStackTrace();
            return null;
        }
    }

    public String getVideoFileName() {
        return videoFileName;
    }

    public JSONObject toJson() {
        try {
            JSONObject object = new JSONObject();
            object.put("video_file_name", videoFileName);
            object.put("crash_uuid", crashUUID);
            object.put("crash_id", crashId);
            object.put("user_id", userId);
            object.put("created_at", createdAt);
            object.put("id", getId());
            return object;
        } catch (JSONException e) {
            e.printStackTrace();
            return null;
        }
    }

    public static class Cache {
        long id;
        String userId;
        String createdAt;
        String crashUUID;
        String crashId;
        String videoFileName;

        public Cache(long id, String userId, String createdAt, String crashId, String crashUUID, String videoFileName) {
            this.id = id;
            this.userId = userId;
            this.crashUUID = crashUUID;
            this.crashId = crashId;
            this.createdAt = createdAt;
            this.videoFileName = videoFileName;
        }


        public boolean isExpired() {
            Date createdAt = getCreatedAt();
            if (createdAt == null) {
                return true;
            }
            Calendar calendar = GregorianCalendar.getInstance();
            Date now = calendar.getTime();
            Date expireTime = new Date(createdAt.getTime() + SolidConstants.VIDEO_EXPIRE_TIME);
            return expireTime.before(now);
        }

        public boolean isReadyToUpload(Context context) {
            if (id < 0) {
                return false;
            }
            return hasVideoFile(context, id, videoFileName) &&
                    (TextUtils.isEmpty(crashUUID) || !TextUtils.isEmpty(crashId));
        }

        public Date getCreatedAt() {
            try {
                SimpleDateFormat format = new SimpleDateFormat(SolidConstants.TIME_FORMAT);
                return format.parse(createdAt);
            } catch (ParseException e) {
                return null;
            }
        }

        public String getUserId() {
            return userId;
        }

        public String getCrashId() {
            return crashId;
        }

        public String getCrashUUID() {
            return crashUUID;
        }

        public long getId() {
            return id;
        }

        public String getVideoFileName() {
            return videoFileName;
        }

        public static VideoMaterial.Cache fromJson(JSONObject object) {
            try {
                long id = object.optLong("id", -1);
                String videoFileName = object.getString("video_file_name");
                String crashUUID = object.optString("crash_uuid");
                String crashId = object.optString("crash_id");
                String userId = object.getString("user_id");
                String createdAt = object.getString("created_at");
                return new VideoMaterial.Cache(id, userId, createdAt, crashId, crashUUID, videoFileName);
            } catch (JSONException e) {
                return null;
            }
        }
    }
}
