package io.solidtech.crash;

import android.app.Activity;
import android.app.ActivityManager;
import android.app.Application;
import android.app.Service;
import android.content.ComponentCallbacks2;
import android.content.Context;
import android.content.Intent;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.content.res.Configuration;
import android.os.Bundle;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Looper;
import android.text.TextUtils;
import android.util.Log;

import com.orm.SugarContext;

import org.json.JSONException;
import org.json.JSONObject;

import java.lang.ref.WeakReference;
import java.util.List;

import io.solidtech.crash.actionlog.ActionLog;
import io.solidtech.crash.actionlog.ActionLogFactory;
import io.solidtech.crash.actionlog.ActionLogManager;
import io.solidtech.crash.entities.CrashReport;
import io.solidtech.crash.environments.ActivityInfo;
import io.solidtech.crash.environments.ActivityStackInfo;
import io.solidtech.crash.utils.AsyncActivityLifecycleTracker;
import io.solidtech.crash.utils.BackgroundTracker;
import io.solidtech.crash.environments.ClientInfo;
import io.solidtech.crash.environments.ConnectivityChangeNotifier;
import io.solidtech.crash.environments.GeolocationInfo;
import io.solidtech.crash.exceptions.SolidRuntimeException;
import io.solidtech.crash.actionlog.UserActionCollector;
import io.solidtech.crash.video.Recorder;

/**
 * Created by vulpes on 2015. 11. 30..
 */
public class SolidClient
        implements Thread.UncaughtExceptionHandler, Application.ActivityLifecycleCallbacks, ComponentCallbacks2,
                   ConnectivityChangeNotifier.SimpleObserver, ConnectivityChangeNotifier.Observer,
                   AnrWatchDog.AnrListener, BackgroundTracker.Observer {


    private static SolidClient sInstance;
    private static SolidUserInfo sDefaultUserInfo;
    private static SolidUserInfo sUserInfo;
    private static ClientInfo sClientInfo;
    private static boolean sIsMainProcess;

    private static final String TAG = SolidClient.class.getSimpleName();

    public static void init(Context context) {
        init(context, null, SolidConfiguration.DEFAUlT);
    }

    public static void init(Context context, SolidConfiguration config) {
        init(context, null, config);
    }

    public static void init(Context context, String apiKey) {
        init(context, apiKey, SolidConfiguration.DEFAUlT);
    }

    public static synchronized void init(Context context, String apiKey, SolidConfiguration config) {

        // setup process info
        Context appContext = context.getApplicationContext();

        long myPid = (long) android.os.Process.myPid();
        List<ActivityManager.RunningAppProcessInfo> runningAppProcesses;
        runningAppProcesses = ((ActivityManager) appContext.getSystemService(Context.ACTIVITY_SERVICE))
                .getRunningAppProcesses();

        if (runningAppProcesses != null) {
            for (ActivityManager.RunningAppProcessInfo runningAppProcessInfo : runningAppProcesses) {
                long pid = (long) runningAppProcessInfo.pid;
                String packageName = appContext.getPackageName();

                if (pid == myPid && packageName.equals(runningAppProcessInfo.processName)) {
                    sIsMainProcess = true;
                    break;
                }
            }
        } else {
            sIsMainProcess = false;
        }

        if (sInstance != null) {

            if (!apiKey.equals(sInstance.getApiKey())) {
                String msg = "SolidClient is already initalized with \"" + sInstance.getApiKey() + "\"";
                throw new SolidRuntimeException(msg);
            }
            Log.w(TAG, "SolidClient already initialized");
            return;
        }

        // create client info
        sClientInfo = ClientInfo.create(context);

        // generate default user info
        sDefaultUserInfo = SolidUserInfo.getAnonymous(context);


        Application app;
        if (context instanceof Activity) {
            app = ((Activity) context).getApplication();
        } else if (context instanceof Service) {
            app = ((Service) context).getApplication();
        } else if (context instanceof Application) {
            app = (Application) context;
        } else {
            throw new SolidRuntimeException("Invalid context. context should be instance of \"Service\" or " +
                    "\"Activity\" or \"Application\"");
        }

        if (apiKey == null) {
            try {
                ApplicationInfo ai = context.getPackageManager()
                                            .getApplicationInfo(app.getPackageName(), PackageManager.GET_META_DATA);
                Bundle bundle = ai.metaData;
                if (bundle != null) {
                    apiKey = bundle.getString(context.getString(R.string.solid_api_key_name));
                }
            } catch (PackageManager.NameNotFoundException e) {
                throw new SolidRuntimeException("Failed to load meta-data", e);
            }
        }
        if (TextUtils.isEmpty(apiKey)) {
            StringBuilder sb = new StringBuilder();
            sb.append("Unable to find solid api key,");
            sb.append("have you declared api key in your AndroidManifest.xml?");
            throw new SolidRuntimeException(sb.toString());
        }

        sInstance = new SolidClient(app, apiKey, config);

        GeolocationInfo.createAsync();
    }

    public static synchronized boolean isInitialized() {
        return sInstance != null;
    }

    public static synchronized void destroy() {
        if (sInstance == null) {
            return;
        }
        sInstance.terminate();
    }

    public static synchronized void setUser(SolidUserInfo userInfo) {
        sUserInfo = userInfo;
    }

    /**
     * Get pre defined user info if exists, or return default anonymous user info.
     * This method never returns null if our client has been successfully initialized.
     */
    public static synchronized SolidUserInfo getUser() {
        if (sUserInfo == null) {
            return sDefaultUserInfo;
        }
        return sUserInfo;
    }

    public static synchronized ClientInfo getClientInfo() {
        return sClientInfo;
    }

    public static void sendCustomException(Throwable throwable) {
        sendCustomException(throwable, null, false);
    }

    public static void sendCustomException(Throwable throwable, boolean withFinishApp) {
        sendCustomException(throwable, null, withFinishApp);
    }

    public static void sendCustomException(Throwable throwable, JSONObject customData) {
        sendCustomException(throwable, customData, false);
    }

    public static synchronized void sendCustomException(final Throwable throwable,
                                                        JSONObject customData,
                                                        final boolean withFinishApp) {

        if (sInstance == null) {
            throw new SolidRuntimeException("SolidClient is not initailized.");
        }

        SolidUserInfo userInfo = getUser();

        // generate report item
        CrashReport.Builder reportBuilder = new CrashReport.Builder(sInstance, userInfo.identifier);

        reportBuilder.setException(Thread.currentThread(), throwable)
                     .appendUser(userInfo)
                     .appendUserCustomData(customData);

        if (sInstance.mActionLogManager != null) {
            reportBuilder.appendActionLogs(sInstance.mActionLogManager.getLogs());
        }

        if (sInstance.mActivityStackInfo != null) {
            reportBuilder.appendActivityStackInfo(sInstance.mActivityStackInfo);
        }

        SolidBackgroundService.remotePushCrash(sInstance.getApplication(), reportBuilder.build());

        if (withFinishApp) {
            Recorder.stopMonitor();

            SolidBackgroundService.remoteAppBackgroundNotify(sInstance.getApplication());
            sInstance.finishApplication(Thread.currentThread(), throwable);
        }
    }

    public static synchronized SolidClient getInstance() {
        return sInstance;
    }

    public static boolean isMainProcess() {
        return sIsMainProcess;
    }

    private final String mApiKey;
    private final Application mApp;
    private final Thread.UncaughtExceptionHandler mParentExceptionHandler;
    private final SolidConfiguration mConfig;
    private ActionLogManager mActionLogManager;
    private ActionLogFactory mActionLogFactory;
    private BackgroundTracker mBackgroundTracker;

    private HandlerThread mActivityLifecycleThread;
    private AsyncActivityLifecycleTracker mActivityLifecycleTracker;

    private ActivityStackInfo mActivityStackInfo;

    public SolidClient(Application app, String apiKey, SolidConfiguration config) {

        if (SolidBackgroundService.isBackgroundProcess(app)) {
            // init sugarcontext
            SugarContext.init(app);

            mApp = app;
            mApiKey = apiKey;
            mConfig = config;

            mParentExceptionHandler = Thread.getDefaultUncaughtExceptionHandler();
            Thread.setDefaultUncaughtExceptionHandler(this);
            return;
        }

        mApp = app;
        mApiKey = apiKey;
        mConfig = config;

        mParentExceptionHandler = Thread.getDefaultUncaughtExceptionHandler();
        Thread.setDefaultUncaughtExceptionHandler(this);

        mActionLogManager = new ActionLogManager();

        // register background/foreground tracker
        mBackgroundTracker = BackgroundTracker.getInstance();
        mBackgroundTracker.registerObserver(this);

        // init infos for diff tracking
        mActionLogFactory = new ActionLogFactory(this);
        mActionLogFactory.startMonitor(mActionLogManager);


        // init activity stack
        mActivityStackInfo = ActivityStackInfo.create(sInstance);


        // listen activity lifecycle for track last activity
        mActivityLifecycleThread = new HandlerThread("SolidClientActivityLifecycleThread");
        mActivityLifecycleThread.start();

        mActivityLifecycleTracker = AsyncActivityLifecycleTracker.getInstance(mApp);
        mActivityLifecycleTracker.addObserver(this, new Handler(mActivityLifecycleThread.getLooper()));



        // listen app memory related callback
        app.registerComponentCallbacks(this);


        // listen network state change
        if (mConfig.useNetworkStateMonitor) {
            listenNetworkStateChange();
        }

        // flush all tracking info
        SolidBackgroundService.remoteFlushAll(mApp);
    }

    public Application getApplication() {
        return mApp;
    }

    public String getString(int resId) {
        return getApplication().getString(resId);
    }

    public String getString(int resId, Object... formatParam) {
        return getApplication().getString(resId, formatParam);
    }

    public SolidConfiguration getConfiguration() {
        return mConfig;
    }

    public String getApiKey() {
        return mApiKey;
    }


    /**
     * listen network state change.
     * Require broadcast receiver registration on AndroidManifest.xml for proper event binding
     */
    public void listenNetworkStateChange() {
        ConnectivityChangeNotifier notifier = ConnectivityChangeNotifier.getInstance(mApp);
        if (mConfig.useDetailNetworkStateMonitor) {
            notifier.addObserver(this);
        }
        notifier.addSimpleObserver(this);
    }


    /**
     * Stop listen network state changes
     */
    public void stopListenNetworkStateChange() {
        ConnectivityChangeNotifier notifier = ConnectivityChangeNotifier.getInstance(mApp);
        if (mConfig.useDetailNetworkStateMonitor) {
            notifier.removeObserver(this);
        }
        notifier.removeSimpleObserver(this);
    }

    /**
     * Destroy solid crash client if exists and terminate all watcher threads
     */
    public void terminate() {

        // terminate SolidVideoService
        if (SolidBackgroundService.isBackgroundProcess(mApp)) {
            // unregister exception handler
            if (mParentExceptionHandler != null) {
                Thread.setDefaultUncaughtExceptionHandler(mParentExceptionHandler);
            } else {
                Thread.setDefaultUncaughtExceptionHandler(null);
            }
            SugarContext.terminate();
            return;
        }

        // unregister app background tracker
        mBackgroundTracker.unregisterObserver(this);

        // unregister memory state tracking
        mApp.unregisterComponentCallbacks(this);

        // unregister exception handler
        if (mParentExceptionHandler != null) {
            Thread.setDefaultUncaughtExceptionHandler(mParentExceptionHandler);
        } else {
            Thread.setDefaultUncaughtExceptionHandler(null);
        }


        // stop listening network state change
        stopListenNetworkStateChange();


        // stop listening ANR error
        AnrWatchDog.stopWatch();


        // stop monitor screen capturn
        Recorder.stopMonitor();

        // stop action log tracking
        mActionLogFactory.stopMonitor();
        mActionLogManager.destroy();

        // stop activity lifecycle tracker
        mActivityLifecycleTracker.removeObserver(this);
        mActivityLifecycleTracker.destroy();

        if (mActivityLifecycleThread != null && mActivityLifecycleThread.isAlive()) {
            mActivityLifecycleThread.interrupt();
        }

        SolidBackgroundService.remoteAppBackgroundNotify(mApp);
    }

    private void finishApplication(Thread thread, Throwable ex) {
        Recorder.stopMonitor();

        try {
            // show custom notify intent
            if (mConfig.customCrashNotifyIntent != null) {
                Intent crashNotifyIntent = mConfig.customCrashNotifyIntent;
                crashNotifyIntent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                mApp.startActivity(crashNotifyIntent);
            } else if (mParentExceptionHandler != null) {
                mParentExceptionHandler.uncaughtException(thread, ex);
                return;
            }

            ex.printStackTrace();

            android.os.Process.killProcess(android.os.Process.myPid());
            System.exit(10);
        } catch (Throwable fataility) {
            fataility.printStackTrace();
            if (mParentExceptionHandler != null) {
                mParentExceptionHandler.uncaughtException(thread, ex);
            } else {
                ex.printStackTrace();
                android.os.Process.killProcess(android.os.Process.myPid());
                System.exit(10);
            }
        }
    }


    /**
     * AnrWatchDog exception handler.
     * This exception runs on AnrWatchDogThread.
     *
     * @param e AnrWatchDog generated exception
     */
    @Override
    public void onAnrException(AnrWatchDog.AnrException e) {
        Recorder.stopMonitor();

        SolidUserInfo userInfo = getUser();
        CrashReport.Builder builder = new CrashReport.Builder(this, userInfo.identifier);

        builder.setException(Looper.getMainLooper().getThread(), e)
               .appendUser(userInfo);

        if (mActivityStackInfo != null) {
            builder.appendActivityStackInfo(mActivityStackInfo);
        }
        if (mActionLogManager != null) {
            builder.appendActionLogs(mActionLogManager.getLogs());
        }

        SolidBackgroundService.remotePushCrash(mApp, builder.build());

        SolidBackgroundService.remoteAppBackgroundNotify(mApp);
        finishApplication(Looper.getMainLooper().getThread(), e);
    }


    /**
     * Overrided Thread uncaught exception handler
     *
     * @param thread exception occurred thread
     * @param ex     occurred exception
     */
    @Override
    public void uncaughtException(final Thread thread, final Throwable ex) {
        ex.printStackTrace();
        Recorder.stopMonitor();
        AnrWatchDog.stopWatch();

        try {
            SolidUserInfo userInfo = getUser();
            CrashReport.Builder builder = new CrashReport.Builder(this, userInfo.identifier);
            builder.setException(thread, ex)
                   .appendUser(userInfo);

            if (mActivityStackInfo != null) {
                builder.appendActivityStackInfo(mActivityStackInfo);
            }
            if (mActionLogManager != null) {
                builder.appendActionLogs(mActionLogManager.getLogs());
            }

            SolidBackgroundService.remotePushCrash(mApp, builder.build());

            SolidBackgroundService.remoteAppBackgroundNotify(mApp);
            finishApplication(thread, ex);

        } catch (Throwable fataility) {
            fataility.printStackTrace();
            if (mParentExceptionHandler != null) {
                mParentExceptionHandler.uncaughtException(thread, ex);
            } else {
                ex.printStackTrace();
            }
        }
    }

    private void mayStartActivityForegroundTasks(Activity activity) {

        // start capture app screen

    }

    /**
     * Handle on activity created.
     *
     * Becasuse this method run on mLifecycleHandlerThread,
     * <b>this method should be thread safe</b>
     *
     * @param activity
     * @param savedInstanceState
     */
    @Override
    public void onActivityCreated(Activity activity, Bundle savedInstanceState) {
        mActivityStackInfo.setActivity(activity, ActivityInfo.STATE_CREATED);
        mActionLogFactory.feedActivityActionLog(mActionLogManager, mActivityStackInfo);

        // listen ANR error
        if (mConfig.useAnr && isMainProcess()) {
            if (!AnrWatchDog.isRunning()) {
                AnrWatchDog.startWatch(mConfig.anrTimeout, this);
            } else if (AnrWatchDog.isPaused()) {
                AnrWatchDog.resumeWatch();
            }
        }
    }

    /**
     * Handle on activity started.
     *
     * Becasuse this method run on mLifecycleHandlerThread,
     * <b>this method should be thread safe</b>
     *
     * @param activity
     */
    @Override
    public void onActivityStarted(Activity activity) {
        mActivityStackInfo.setActivity(activity, ActivityInfo.STATE_STARTED);
        mActionLogFactory.feedActivityActionLog(mActionLogManager, mActivityStackInfo);

        // listen ANR error
        if (mConfig.useAnr && isMainProcess()) {
            if (!AnrWatchDog.isRunning()) {
                AnrWatchDog.startWatch(mConfig.anrTimeout, this);
            } else if (AnrWatchDog.isPaused()) {
                AnrWatchDog.resumeWatch();
            }
        }
    }

    /**
     * Handle on activity resumed.
     *
     * Becasuse this method run on mLifecycleHandlerThread,
     * <b>this method should be thread safe</b>
     *
     * @param activity
     */
    @Override
    public void onActivityResumed(Activity activity) {
        // UserActionCollection should bind after window being set
        UserActionCollector.bind(activity, mActionLogManager);

        mActivityStackInfo.setActivity(activity, ActivityInfo.STATE_RESUMED);
        mActionLogFactory.feedActivityActionLog(mActionLogManager, mActivityStackInfo);

        // stop background transition timer because app is in foreground
        mBackgroundTracker.stopActivityTransitionTimer();

        if (mConfig.useVideo) {
            Recorder.startMonitor(this, activity);
        }

        // listen ANR error
        if (mConfig.useAnr && isMainProcess()) {
            if (!AnrWatchDog.isRunning()) {
                AnrWatchDog.startWatch(mConfig.anrTimeout, this);
            } else if (AnrWatchDog.isPaused()) {
                AnrWatchDog.resumeWatch();
            }
        }
    }

    /**
     * Handle on activity paused.
     *
     * Becasuse this method run on mLifecycleHandlerThread,
     * <b>this method should be thread safe</b>
     *
     * @param activity
     */
    @Override
    public void onActivityPaused(Activity activity) {
        // Unbind UserActionCollector
        UserActionCollector.unbind(activity);

        mActivityStackInfo.setActivity(activity, ActivityInfo.STATE_PAUSED);
        mActionLogFactory.feedActivityActionLog(mActionLogManager, mActivityStackInfo);

        // start background transition timer because app can be in background
        mBackgroundTracker.startActivityTransitionTimer();

        // listen ANR error
        if (mConfig.useAnr && isMainProcess() && AnrWatchDog.isRunning() && !AnrWatchDog.isPaused()) {
            AnrWatchDog.pauseWatch();
        }
    }

    /**
     * Handle on activity stopped.
     *
     * Becasuse this method run on mLifecycleHandlerThread,
     * <b>this method should be thread safe</b>
     *
     * @param activity
     */
    @Override
    public void onActivityStopped(Activity activity) {
        mActivityStackInfo.setActivity(activity, ActivityInfo.STATE_STOPPED);
        mActionLogFactory.feedActivityActionLog(mActionLogManager, mActivityStackInfo);
    }

    /**
     * Handle on activity's save instance state.
     *
     * Becasuse this method run on mLifecycleHandlerThread,
     * <b>this method should be thread safe</b>
     *
     * @param activity
     */
    @Override
    public void onActivitySaveInstanceState(Activity activity, Bundle outState) {
        mActivityStackInfo.setActivity(activity, ActivityInfo.STATE_SAVE_INSTANCE_STATE);
        mActionLogFactory.feedActivityActionLog(mActionLogManager, mActivityStackInfo);
    }

    /**
     * Handle on activity destroyed.
     *
     * Becasuse this method run on mLifecycleHandlerThread,
     * <b>this method should be thread safe</b>
     *
     * @param activity
     */
    @Override
    public void onActivityDestroyed(Activity activity) {
        mActivityStackInfo.setActivity(activity, ActivityInfo.STATE_DESTROYED);
        mActionLogFactory.feedActivityActionLog(mActionLogManager, mActivityStackInfo);
    }

    @Override
    public void onTrimMemory(int level) {
        String state = "unknown";
        switch (level) {
            case TRIM_MEMORY_BACKGROUND:
                state = "background";
                break;
            case TRIM_MEMORY_COMPLETE:
                state = "complete";
                break;
            case TRIM_MEMORY_MODERATE:
                state = "moderate";
                break;
            case TRIM_MEMORY_RUNNING_CRITICAL:
                state = "running_critical";
                break;
            case TRIM_MEMORY_RUNNING_LOW:
                state = "running_low";
                break;
            case TRIM_MEMORY_RUNNING_MODERATE:
                state = "running_moderate";
                break;
            case TRIM_MEMORY_UI_HIDDEN:
                state = "ui_hidden";
                break;
        }
        //Log.d(TAG, "onTrimMemory: " + state);
    }

    @Override
    public void onConfigurationChanged(Configuration newConfig) {
        //Log.d(TAG, "onConfigurationChanged");
    }

    @Override
    public void onLowMemory() {
        //Log.d(TAG, "onLowMemory");
    }

    @Override
    public void onInternetConnected(boolean isWifi) {
        // renew cache for public ip renew
        GeolocationInfo.clearCache();
        GeolocationInfo.createAsync();

        // flush all tracking info
        SolidBackgroundService.remoteFlushAll(mApp);

        if (mConfig.useDetailNetworkStateMonitor) {
            return;
        }

        mActionLogFactory.feedNetworkActionLog(mActionLogManager);
    }

    @Override
    public void onInternetDisconnected() {
        if (mConfig.useDetailNetworkStateMonitor) {
            return;
        }
        mActionLogFactory.feedNetworkActionLog(mActionLogManager);
    }

    @Override
    public void onInternetConnectTypeChanged(boolean isWifi) {
        // renew cache for public ip renew
        GeolocationInfo.clearCache();
        GeolocationInfo.createAsync();

        // flush all tracking info
        SolidBackgroundService.remoteFlushAll(mApp);

        if (mConfig.useDetailNetworkStateMonitor) {
            return;
        }
        mActionLogFactory.feedNetworkActionLog(mActionLogManager);
    }

    @Override
    public void onNetworkStateChanged(List<android.net.NetworkInfo> infoList) {
        mActionLogFactory.feedNetworkActionLog(mActionLogManager);
    }

    /**
     * push application diff when app become foreground
     */
    @Override
    public void onAppBecomeForeground() {
        try {
            JSONObject data = new JSONObject();
            data.put("app_state", new JSONObject().put("prev", "background").put("curr", "foreground"));
            ActionLog actionLog = new ActionLog(ActionLog.Type.APP_DIFF, data);
            mActionLogManager.push(actionLog);
        } catch (JSONException e) {
            // not reach
        }

        if (mConfig.useVideo) {
            Recorder.resumeMonitor();
        }

        // notify to service
        SolidBackgroundService.remoteAppForegroundNotify(mApp);
    }

    /**
     * push application diff when app become background
     */
    @Override
    public void onAppBecomeBackground() {
        try {
            JSONObject data = new JSONObject();
            data.put("app_state", new JSONObject().put("prev", "foreground").put("curr", "background"));
            ActionLog actionLog = new ActionLog(ActionLog.Type.APP_DIFF, data);
            mActionLogManager.push(actionLog);
        } catch (JSONException e) {
            // not reach
        }

        if (mConfig.useVideo) {
            Recorder.pauseMonitor();
            SolidBackgroundService.remoteVideoCreate(mApp);
        }

        // notify to service
        SolidBackgroundService.remoteAppBackgroundNotify(mApp);
    }
}


