package io.solidtech.crash.utils;

import android.os.Handler;
import android.os.Looper;

import java.util.ArrayList;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;

/**
 * Created by vulpes on 16. 1. 4..
 */
public class BackgroundTracker {

    public interface Observer {
        void onAppBecomeForeground();
        void onAppBecomeBackground();
    }

    private static final long MAX_ACTIVITY_TRANSITION_TIME_MS = 2000;
    private static final String TAG = BackgroundTracker.class.getSimpleName();
    private static final boolean VERBOSE = false;
    private static final DebugUtils.DebugLogger sLogger = new DebugUtils.DebugLogger(TAG, VERBOSE);

    private static BackgroundTracker sInstance;

    public synchronized static BackgroundTracker getInstance() {
        if (sInstance == null) {
            sInstance = new BackgroundTracker();
        }
        return sInstance;
    }


    private final Handler mHandler;

    private boolean mWasAppInBackground = false; // assume that app is in foreground when initialize background tracker
    private Timer mActivityTransitionTimer;
    private TimerTask mActivityTransitionTimerTask;
    private List<Observer> mObservers = new ArrayList<>();

    private BackgroundTracker() {
        mHandler = new Handler(Looper.getMainLooper());
    }

    public boolean registerObserver(Observer observer) {
        synchronized (mObservers) {
            return mObservers.add(observer);
        }
    }

    public boolean unregisterObserver(Observer observer) {
        synchronized (mObservers) {
            return mObservers.remove(observer);
        }
    }

    public void startActivityTransitionTimer() {
        mActivityTransitionTimer = new Timer();
        mActivityTransitionTimerTask = new TimerTask() {
            public void run() {
                if (!mWasAppInBackground) {
                    synchronized (this) {
                        if (!mWasAppInBackground) {
                            mWasAppInBackground = true;
                            onAppBecomeBackground();
                        }
                    }
                }
            }
        };

        mActivityTransitionTimer.schedule(mActivityTransitionTimerTask,
                MAX_ACTIVITY_TRANSITION_TIME_MS);
    }

    public void stopActivityTransitionTimer() {
        if (this.mActivityTransitionTimerTask != null) {
            this.mActivityTransitionTimerTask.cancel();
        }

        if (this.mActivityTransitionTimer != null) {
            this.mActivityTransitionTimer.cancel();
        }

        if (mWasAppInBackground) {
            synchronized (this) {
                if (mWasAppInBackground) {
                    mWasAppInBackground = false;
                    onAppBecomeForeground();
                }
            }
        }
    }

    public synchronized boolean wasAppInBackground() {
        return mWasAppInBackground;
    }

    private void onAppBecomeBackground() {
        sLogger.d("app is background");

        for(final Observer observer : new ArrayList<>(mObservers)) {
            mHandler.post(new Runnable() {
                @Override
                public void run() {
                    observer.onAppBecomeBackground();
                }
            });
        }
    }

    private void onAppBecomeForeground() {
        sLogger.d("app is foreground");

        for(final Observer observer : new ArrayList<>(mObservers)) {
            mHandler.post(new Runnable() {
                @Override
                public void run() {
                    observer.onAppBecomeForeground();
                }
            });
        }
    }
}
