/*
 * Copyright (C) 2016 Dropbeat, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.solidtech.crash.network;

import org.apache.http.Header;
import org.apache.http.HttpEntity;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * HTTP 1.x parser.
 */
public class HttpParser {
     public static Map<String, List<String>> getQueryParams(String url) {
        Map<String, List<String>> params = new HashMap<String, List<String>>();
        String[] elems = url.split("\\?");
        if (elems.length <= 1) {
            return params;
        }

        try {
            for (String param : elems[1].split("&")) {
                String[] pair = param.split("=");
                String key = URLDecoder.decode(pair[0], "UTF-8");
                String value = "";
                if (pair.length > 1) {
                    value = URLDecoder.decode(pair[1], "UTF-8");
                }

                List<String> values = params.get(key);
                if (values == null) {
                    values = new ArrayList<String>();
                    params.put(key, values);
                }
                values.add(value);
            }
            return params;
        } catch (UnsupportedEncodingException ex) {
            return params;
        }
    }

    /**
     * Calculate total bytes sent from HTTP request.
     * @param method
     * @param url
     * @param headers
     * @param body
     * @return
     */
    public static long getBytesSent(
            String method, String url, Map<String, String> headers, String body) {
        // GET /hello HTTP/1.1
        long bytes = 0;
        bytes += method.length() + 1;
        bytes += url.length() + 1;
        bytes += "HTTP/1.1".length() + 2;
        bytes += getHeaderBytes(headers);
        bytes += body.length();
        return bytes;
    }

    /**
     * Caclulate total bytes from HTTP response.
     * Because getting reasonPhrase sucks, let's regard it as 6 bytes.
     * TODO: Calculate reasonPhrase exactly. (although it's not important)
     * @param headers
     * @param contentLength
     * @return
     */
    public static long getBytesReceived(Map<String, String> headers, long contentLength) {
        // HTTP/1.1 400 Bad Request
        long bytes = 0;
        // 1 byte + 3 bytes (status code)
        bytes += "HTTP/1.1".length() + 4;
        long reasonPhraseLength = 6;
        bytes += reasonPhraseLength + 2;
        bytes += getHeaderBytes(headers);
        bytes += contentLength;
        return bytes;
    }

    /**
     * NOTE that this includes last 4 bytes of headers (\r\n\r\n).
     * @param headers
     * @return
     */
    private static long getHeaderBytes(Map<String, String> headers) {
        long bytes = 0;
        // Headers
        for (String key : headers.keySet()) {
            bytes += key.length();
            // According to RFC 7230,
            // The field value MAY be preceded by any amount of LWS, though a single SP is preferred.
            // We regard it as 2 bytes (': ').
            bytes += 2;
            bytes += headers.get(key).length();
            // 2 bytes for CRLF
            bytes += 2;
        }
        // End of headers.
        bytes += 2;
        return bytes;
    }

    public static Map<String, String> httpHeaderToMap(Header[] headers) {
        Map<String, String> headerMap = new HashMap<String, String>();
        for (Header header : headers) {
            headerMap.put(header.getName(), header.getValue());
        }
        return headerMap;
    }

    /**
     * Parse HTTP entity to string.
     * This method returns first 100 byte of response body because reading whole entity
     * into memory is dangerous.
     * @param entity
     * @return
     */
    public static String readHttpEntity(HttpEntity entity) {
        try {
            final int LIMIT = 100;
            StringBuilder body = new StringBuilder();
            BufferedReader reader =
                    new BufferedReader(new InputStreamReader(entity.getContent()));
            String line;
            while ((line = reader.readLine()) != null) {
                if (body.toString().length() > LIMIT) {
                    break;
                }
                body.append(line);
            }
            return body.toString().substring(0, Math.min(body.length(), LIMIT));
        } catch (IOException e) {
            return null;
        }

    }
}
