package io.solidtech.crash.environments;

import android.net.ConnectivityManager;

import org.json.JSONException;
import org.json.JSONObject;

import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import io.solidtech.crash.SolidClient;

/**
 * Created by vulpes on 2015. 12. 1..
 */
public class NetworkInfo {

    private static final String TAG = NetworkInfo.class.getSimpleName();

    public static NetworkInfo create(SolidClient client) {
        ConnectivityChangeNotifier notifier = ConnectivityChangeNotifier.getInstance(client.getApplication());
        List<android.net.NetworkInfo> netInfoList = new ArrayList<>(notifier.getLastNetworkInfoList());

        StringBuilder readableStateBuilder = new StringBuilder();
        if (notifier.isConnected()) {
            readableStateBuilder.append("Connected");
            readableStateBuilder.append(" - ");
            if (notifier.isWifiConnected()) {
                readableStateBuilder.append("Wifi");
            } else {
                readableStateBuilder.append("Mobile");
            }
        } else {
            readableStateBuilder.append("Disconnected");
        }

        NetworkInfo info = new NetworkInfo(readableStateBuilder.toString());
        for (android.net.NetworkInfo netInfo : netInfoList) {

            if (!client.getConfiguration().useDetailNetworkStateMonitor &&
                    netInfo.getType() != ConnectivityManager.TYPE_MOBILE &&
                    netInfo.getType() != ConnectivityManager.TYPE_WIFI) {
                continue;
            }

            Item item = new Item();
            item.isRoaming = netInfo.isRoaming();
            item.isAvailable = netInfo.isAvailable();
            item.isFailover = netInfo.isFailover();
            item.isConnected = netInfo.isConnected();
            item.isConnectedOrConnecting = netInfo.isConnectedOrConnecting();
            item.type = netInfo.getType();
            item.typeName = netInfo.getTypeName();
            item.subType = netInfo.getSubtype();
            item.subTypeName = netInfo.getSubtypeName();
            item.state = netInfo.getState().name();
            item.detailState = netInfo.getDetailedState().name();
            item.extraInfo = netInfo.getExtraInfo();
            item.reason = netInfo.getReason();

            info.detail.put(netInfo.getTypeName(), item);

        }

        return info;
    }

    private static String getLocalIpAddress() {
        try {
            List<NetworkInterface> interfaces = Collections.list(NetworkInterface.getNetworkInterfaces());
            for (NetworkInterface intf : interfaces) {
                List<InetAddress> addrs = Collections.list(intf.getInetAddresses());
                for (InetAddress addr : addrs) {
                    if (addr.isLoopbackAddress()) {
                        continue;
                    }
                    String sAddr = addr.getHostAddress();
                    boolean isIPv4 = sAddr.indexOf(':') < 0;
                    if (isIPv4) {
                        return sAddr;
                    }
                }
            }
        } catch (SocketException e) {
            // do nothing
        }

        return null;
    }

    private final String readableState;
    private final String ipAddress;
    private final Map<String, Item> detail = new HashMap<>();

    private NetworkInfo(String readableState) {
        this.readableState = readableState;
        this.ipAddress = getLocalIpAddress();
    }

    public JSONObject toJSONObject() {
        try {
            JSONObject obj = new JSONObject();
            obj.put("readable_state", readableState);
            obj.put("ip_address", ipAddress);

            JSONObject detailObj = new JSONObject();
            for (String key : detail.keySet()) {
                detailObj.put(key, detail.get(key).toJSONObject());
            }
            obj.put("detail", detailObj);
            return obj;
        } catch (JSONException e) {
            return null;
        }
    }

    private static class Item {
        boolean isRoaming;
        boolean isFailover;
        boolean isConnectedOrConnecting;
        boolean isConnected;
        boolean isAvailable;
        String typeName;
        int type;
        int subType;
        String subTypeName;
        String reason;
        String extraInfo;
        String state;
        String detailState;

        public JSONObject toJSONObject() throws JSONException {
            JSONObject obj = new JSONObject()
                    .put("is_roaming", isRoaming)
                    .put("is_failover", isFailover)
                    .put("is_connected_or_connecting", isConnectedOrConnecting)
                    .put("is_connected", isConnected)
                    .put("is_available", isAvailable)
                    .put("type_name", typeName)
                    .put("type", type)
                    .put("sub_type", subType)
                    .put("sub_type_name", subTypeName)
                    .put("reason", reason)
                    .put("extra_info", extraInfo)
                    .put("state", state)
                    .put("detail_state", detailState);

            return obj;
        }
    }
}
