package io.solidtech.crash.environments;

import com.squareup.okhttp.Call;
import com.squareup.okhttp.Callback;
import com.squareup.okhttp.Request;
import com.squareup.okhttp.Response;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;

import io.solidtech.crash.SolidConstants;
import io.solidtech.crash.network.BasicRequests;

/**
 * Created by vulpes on 2015. 12. 3..
 */
public class GeolocationInfo {

    public interface OnGeolocationInfoCreateListener {
        void onGeolocationInfoCreated(GeolocationInfo info);
    }

    private static GeolocationInfo sInstance;
    private static Call sOnTheFlyCall;

    public static GeolocationInfo getCachedInstance() {
        return sInstance;
    }

    public static synchronized void clearCache() {
        if (sOnTheFlyCall != null) {
            sOnTheFlyCall.cancel();
            sOnTheFlyCall = null;
        }
        sInstance = null;
    }

    /**
     * This function blocks thread due to fetch information from remote web server.
     * So this should not be called on main thread.
     */
    public static void createAsync() {
        createAsync(null);
    }

    /**
     * This function blocks thread due to fetch information from remote web server.
     * So this should not be called on main thread.
     */
    public static synchronized void createAsync(final OnGeolocationInfoCreateListener listener) {
        if (sOnTheFlyCall != null) {
            sOnTheFlyCall.cancel();
        }
        if (sInstance != null) {
            if (listener != null) {
                listener.onGeolocationInfoCreated(sInstance);
            }
        }

        Call call = BasicRequests.buildGetCall(SolidConstants.GEOLOCATION_URL, null);
        sOnTheFlyCall = call;

        call.enqueue(new Callback() {
            @Override
            public void onFailure(Request request, IOException e) {
                synchronized (GeolocationInfo.class) {
                    if (sOnTheFlyCall != null) {
                        sOnTheFlyCall = null;
                    }
                }
            }

            @Override
            public void onResponse(Response response) throws IOException {
                synchronized (GeolocationInfo.class) {
                    if (sOnTheFlyCall != null) {
                        sOnTheFlyCall = null;
                    }
                    try {
                        sInstance = GeolocationInfo.fromJson(response.body().string());
                        if (listener != null) {
                            listener.onGeolocationInfoCreated(sInstance);
                        }
                    } catch (JSONException e) {

                    }
                }
            }
        });
    }

    private static synchronized GeolocationInfo createSync() {
        if (sOnTheFlyCall != null) {
            sOnTheFlyCall.cancel();
        }
        if (sInstance != null) {
            return sInstance;
        }

        try {
            Call call = BasicRequests.buildGetCall(SolidConstants.GEOLOCATION_URL, null);
            sOnTheFlyCall = call;
            Response response = call.execute();
            sOnTheFlyCall = null;

            sInstance = GeolocationInfo.fromJson(response.body().string());
        } catch (Throwable throwable) {
            // do nothing
        }
        return sInstance;
    }

    public static GeolocationInfo fromJson(String obj) throws JSONException {
        JSONObject object = new JSONObject(obj);
        return fromJson(object);
    }

    public static GeolocationInfo fromJson(JSONObject obj) throws JSONException {
        GeolocationInfo info = new GeolocationInfo();
        info.ip = obj.optString("ip");
        info.city = obj.optString("city");
        info.countryCode = obj.optString("country_code");
        info.countryName = obj.optString("country_name");
        info.regionCode = obj.optString("region_code");
        info.regionName = obj.optString("region_name");
        info.latitude = (float) obj.optDouble("latitude");
        info.longitude = (float) obj.optDouble("longitude");
        info.zipCode = obj.optString("zip_code");
        info.timeZone = obj.optString("time_zone");

        return info;
    }

    private String ip;
    private String countryCode;
    private String countryName;
    private String regionCode;
    private String regionName;
    private String city;
    private String zipCode;
    private String timeZone;
    private float latitude;
    private float longitude;


    public JSONObject toJSONObject() {
        try {
            return new JSONObject().put("ip", ip)
                                   .put("country_code", countryCode)
                                   .put("country_name", countryName)
                                   .put("region_code", regionCode)
                                   .put("region_name", regionName)
                                   .put("city", city)
                                   .put("zip_code", zipCode)
                                   .put("time_zone", timeZone)
                                   .put("latitude", latitude)
                                   .put("longitude", longitude);
        } catch (JSONException e) {
            return null;
        }
    }
}
