package io.solidtech.crash.environments;

import android.os.Looper;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;

import io.solidtech.crash.AnrWatchDog;
import io.solidtech.crash.SolidClient;
import io.solidtech.crash.SolidConstants;
import io.solidtech.crash.SolidThread;

/**
 * Created by vulpes on 2015. 12. 2..
 */
public class ExceptionInfo {
    public static ExceptionInfo create(SolidClient client, Thread thread, Throwable e) {
        ExceptionInfo info = new ExceptionInfo(client, thread, e);
        return info;
    }

    private List<StackTraceInfo> mAllStacktraces = new ArrayList<>();

    private ExceptionInfo(SolidClient client, Thread thread, Throwable e) {

        final String crucialClassName = e.getClass().getName();
        final String crucialMessage = e.getMessage();
        StackTraceElement[] crucialStacktrace = e.getStackTrace();
        if (e instanceof AnrWatchDog.AnrException || thread == null) {
            thread = Looper.getMainLooper().getThread();
            crucialStacktrace = thread.getStackTrace();
        }
        final StackTraceInfo crucialInfo = new StackTraceInfo(
                thread.getName(),
                crucialClassName,
                crucialMessage,
                e.getCause(),
                crucialStacktrace);
        mAllStacktraces.add(crucialInfo);

        final Map<Thread, StackTraceElement[]> all = Thread.getAllStackTraces();
        for (Thread t : all.keySet()) {
            if (SolidConstants.MUTE_SOLID_ERROR &&
                    SolidThread.class.isAssignableFrom(t.getClass()) || t == thread) {
                continue;
            }
            StackTraceElement[] stackTraceElements = t.getStackTrace();
            if (stackTraceElements == null || stackTraceElements.length == 0) {
                continue;
            }
            mAllStacktraces.add(new StackTraceInfo(t.getName(), stackTraceElements));
        }

        final String mainThreadName = Looper.getMainLooper().getThread().getName();
        Collections.sort(mAllStacktraces, new Comparator<StackTraceInfo>() {
            @Override
            public int compare(StackTraceInfo lhs, StackTraceInfo rhs) {
                if (lhs.name.equals(rhs.name)) {
                    return 0;
                }
                if (rhs == crucialInfo) {
                    return 1;
                }
                if (lhs == crucialInfo) {
                    return -1;
                }
                if (rhs.name.equals(mainThreadName)) {
                    return 1;
                }
                if (lhs.name.equals(mainThreadName)) {
                    return -1;
                }
                return lhs.name.compareTo(rhs.name);
            }
        });
    }

    public JSONArray toJSONArray() {
        JSONArray array = new JSONArray();
        for (StackTraceInfo info : mAllStacktraces) {
            array.put(info.toJSONObject());
        }
        return array;
    }

    private static class StackTraceInfo {
        private String name;
        private String className;
        private String message;
        private StackTraceInfo innerError;
        private List<StackTraceItem> stacktrace = new ArrayList<>();

        public StackTraceInfo(String name, StackTraceElement[] items) {
            this(name, null, null, items);
        }

        public StackTraceInfo(String name, String className, String message, StackTraceElement[] items) {
            this(name, className, message, null, items);
        }

        public StackTraceInfo(String name,
                              String className,
                              String message,
                              Throwable throwable,
                              StackTraceElement[] items) {
            this.name = name;
            this.className = className;
            this.message = message;
            if (throwable != null) {
                this.innerError = new StackTraceInfo(name, throwable.getClass().getName(), throwable.getMessage(),
                        throwable.getCause(), throwable.getStackTrace());
            }
            for (StackTraceElement item : items) {
                stacktrace.add(new StackTraceItem(item));
            }
        }

        public JSONObject toJSONObject() {
            try {
                JSONArray stacktraceArray = new JSONArray();
                for (StackTraceItem item : stacktrace) {
                    stacktraceArray.put(item.toJSONObject());
                }
                JSONObject obj = new JSONObject().put("name", name)
                        .put("class_name", className)
                        .put("message", message)
                        .put("stacktrace", stacktraceArray);

                if (innerError != null) {
                    obj.put("inner_error", innerError.toJSONObject());
                }
                return obj;
            } catch (JSONException e) {
                return null;
            }
        }
    }

    private static class StackTraceItem {
        private final String className;
        private final String fileName;
        private final String methodName;
        private final int lineNumber;

        public StackTraceItem(StackTraceElement el) {
            className = el.getClassName();
            fileName = el.getFileName();
            lineNumber = el.getLineNumber();
            methodName = el.getMethodName();
        }

        public JSONObject toJSONObject() throws JSONException {
            return new JSONObject().put("class_name", className)
                    .put("file_name", fileName)
                    .put("method_name", methodName)
                    .put("line_number", lineNumber);
        }
    }
}
