package io.solidtech.crash.environments;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.net.ConnectivityManager;
import android.net.Network;
import android.net.NetworkInfo;
import android.os.Build;
import android.util.Log;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import io.solidtech.crash.utils.DebugUtils;

public class ConnectivityChangeNotifier {

    public interface SimpleObserver {
        void onInternetConnected(boolean isWifi);

        void onInternetDisconnected();

        void onInternetConnectTypeChanged(boolean isWifi);
    }

    public interface Observer {
        void onNetworkStateChanged(List<NetworkInfo> infoList);
    }

    private static final String TAG = ConnectivityChangeNotifier.class.getSimpleName();
    private static final boolean VERBOSE = false;
    private static final DebugUtils.DebugLogger sLogger = DebugUtils.createDebugLogger(TAG, VERBOSE);
    private static ConnectivityChangeNotifier sInstance;

    public static final int STATE_WIFI = 1;
    public static final int STATE_MOBILE = 2;

    public static ConnectivityChangeNotifier getInstance(Context context) {
        if (sInstance != null) {
            sInstance.updateNetworkState();
            return sInstance;
        }
        synchronized (ConnectivityChangeNotifier.class) {
            if (sInstance == null) {
                sInstance = new ConnectivityChangeNotifier(context);
            }
            return sInstance;
        }
    }

    private List<SimpleObserver> mSimpleObservers = new ArrayList<>();
    private List<Observer> mObservers = new ArrayList<>();
    private int mConnectState;
    private ConnectivityManager mConnManager;
    private List<NetworkInfo> mLastNetworkInfos = new ArrayList<>();

    private ConnectivityChangeNotifier(Context context) {
        mConnManager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
        // set initial network state
        updateNetworkState();
    }

    public boolean addObserver(Observer ob) {
        synchronized (mObservers) {
            return mObservers.add(ob);
        }
    }

    public boolean removeObserver(Observer ob) {
        synchronized (mObservers) {
            return mObservers.remove(ob);
        }
    }

    public boolean addSimpleObserver(SimpleObserver ob) {
        synchronized (mSimpleObservers) {
            return mSimpleObservers.add(ob);
        }
    }

    public boolean removeSimpleObserver(SimpleObserver ob) {
        synchronized (mSimpleObservers) {
            return mSimpleObservers.remove(ob);
        }
    }

    public synchronized boolean isConnected() {
        return mConnectState > 0;
    }

    public synchronized boolean isWifiConnected() {
        return (mConnectState & STATE_WIFI) > 0;
    }

    public synchronized boolean isMobileConnected() {
        return (mConnectState & STATE_MOBILE) > 0;
    }

    public synchronized List<NetworkInfo> getLastNetworkInfoList() {
        return mLastNetworkInfos;
    }

    private synchronized int getConnectState() {
        return mConnectState;
    }

    private void notifyConnected(boolean isWifi) {
        sLogger.d("notify internet connected");
        for (SimpleObserver ob : new ArrayList<>(mSimpleObservers)) {
            ob.onInternetConnected(isWifi);
        }
    }

    private void notifyDisconnected() {
        sLogger.d("notify internet disconnected");
        for (SimpleObserver ob : new ArrayList<>(mSimpleObservers)) {
            ob.onInternetDisconnected();
        }
    }

    private void notifyConnectTypeChanged() {
        sLogger.d("notify internet connect type changed");
        for (SimpleObserver ob : new ArrayList<>(mSimpleObservers)) {
            ob.onInternetConnectTypeChanged(isWifiConnected());
        }
    }

    private synchronized void setMobileConnected(boolean connected) {
        boolean isMobilePrevConnected = isMobileConnected();
        if (isMobilePrevConnected && !connected) {
            mConnectState -= STATE_MOBILE;
        } else if (!isMobilePrevConnected && connected) {
            mConnectState += STATE_MOBILE;
        }
    }

    private synchronized void setWifiConnected(boolean connected) {
        boolean isWifiPrevConnected = isWifiConnected();
        if (isWifiPrevConnected && !connected) {
            mConnectState -= STATE_WIFI;
        } else if (!isWifiPrevConnected && connected) {
            mConnectState += STATE_WIFI;
        }
    }

    private synchronized void updateNetworkState() {
        List<NetworkInfo> netInfos = new ArrayList<>();
        if (Build.VERSION.SDK_INT >= 21) {
            Network[] networks = mConnManager.getAllNetworks();
            if (networks != null) {
                for (Network network : networks) {
                    NetworkInfo info = mConnManager.getNetworkInfo(network);
                    if (info != null) {
                        netInfos.add(info);
                    }
                }
            }
        } else {
            NetworkInfo[] infos = mConnManager.getAllNetworkInfo();
            if (infos != null) {
                netInfos.addAll(Arrays.asList(infos));
            }
        }
        if (netInfos.size() == 0) {
            return;
        }

        int prevConnectedState = getConnectState();

        mLastNetworkInfos.clear();
        for (NetworkInfo netInfo : netInfos) {
            switch (netInfo.getType()) {
                case ConnectivityManager.TYPE_MOBILE:
                    setMobileConnected(netInfo.isConnected());
                    break;
                case ConnectivityManager.TYPE_WIFI:
                    setWifiConnected(netInfo.isConnected());
                    break;
            }
            mLastNetworkInfos.add(netInfo);
        }
        int currConnectedState = getConnectState();
        if (prevConnectedState != currConnectedState) {
            if (currConnectedState == 0) {
                notifyDisconnected();
            } else if (prevConnectedState == 0) {
                notifyConnected(isWifiConnected());
            } else {
                notifyConnectTypeChanged();
            }
        }
    }

    public static class Receiver extends BroadcastReceiver {

        @Override
        public void onReceive(Context context, Intent intent) {
            ConnectivityChangeNotifier manager = getInstance(context);
            manager.updateNetworkState();
            for (Observer ob : new ArrayList<>(manager.mObservers)) {
                ob.onNetworkStateChanged(manager.getLastNetworkInfoList());
            }
        }
    }
}
