package io.solidtech.crash.actionlog;

import android.annotation.TargetApi;
import android.app.Activity;
import android.os.Build;
import android.support.annotation.Nullable;
import android.view.ActionMode;
import android.view.GestureDetector;
import android.view.KeyEvent;
import android.view.Menu;
import android.view.MenuItem;
import android.view.MotionEvent;
import android.view.SearchEvent;
import android.view.View;
import android.view.Window;
import android.view.WindowManager;
import android.view.accessibility.AccessibilityEvent;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.Map;

import io.solidtech.crash.utils.DebugUtils;

/**
 * Created by vulpes on 15. 12. 30..
 */
public class UserActionCollector implements Window.Callback, GestureDetector
        .OnDoubleTapListener, GestureDetector.OnGestureListener {

    public static Map<Activity, UserActionCollector> sCollectors = new HashMap<>();

    private static final String TAG = UserActionCollector.class.getSimpleName();
    private static final boolean VERBOSE = false;
    private static final DebugUtils.DebugLogger sLogger = new DebugUtils.DebugLogger(TAG, VERBOSE);

    // Symbolic names of all metakeys in bit order from least significant to most significant.
    // Accordingly there are exactly 32 values in this table.
    private static final String[] META_SYMBOLIC_NAMES = new String[] {
            "META_SHIFT_ON",
            "META_ALT_ON",
            "META_SYM_ON",
            "META_FUNCTION_ON",
            "META_ALT_LEFT_ON",
            "META_ALT_RIGHT_ON",
            "META_SHIFT_LEFT_ON",
            "META_SHIFT_RIGHT_ON",
            "META_CAP_LOCKED",
            "META_ALT_LOCKED",
            "META_SYM_LOCKED",
            "0x00000800",
            "META_CTRL_ON",
            "META_CTRL_LEFT_ON",
            "META_CTRL_RIGHT_ON",
            "0x00008000",
            "META_META_ON",
            "META_META_LEFT_ON",
            "META_META_RIGHT_ON",
            "0x00080000",
            "META_CAPS_LOCK_ON",
            "META_NUM_LOCK_ON",
            "META_SCROLL_LOCK_ON",
            "0x00800000",
            "0x01000000",
            "0x02000000",
            "0x04000000",
            "0x08000000",
            "0x10000000",
            "0x20000000",
            "0x40000000",
            "0x80000000",
    };

    // Symbolic names of all button states in bit order from least significant
    // to most significant.
    private static final String[] BUTTON_SYMBOLIC_NAMES = new String[] {
            "BUTTON_PRIMARY",
            "BUTTON_SECONDARY",
            "BUTTON_TERTIARY",
            "BUTTON_BACK",
            "BUTTON_FORWARD",
            "BUTTON_STYLUS_PRIMARY",
            "BUTTON_STYLUS_SECONDARY",
            "0x00000080",
            "0x00000100",
            "0x00000200",
            "0x00000400",
            "0x00000800",
            "0x00001000",
            "0x00002000",
            "0x00004000",
            "0x00008000",
            "0x00010000",
            "0x00020000",
            "0x00040000",
            "0x00080000",
            "0x00100000",
            "0x00200000",
            "0x00400000",
            "0x00800000",
            "0x01000000",
            "0x02000000",
            "0x04000000",
            "0x08000000",
            "0x10000000",
            "0x20000000",
            "0x40000000",
            "0x80000000",
    };

    public static UserActionCollector bind(Activity activity, ActionLogManager manager) {
        synchronized (sCollectors) {
            UserActionCollector collector = sCollectors.get(activity);
            if (collector == null) {
                Window window = activity.getWindow();
                if (window == null) {
                    return null;
                }
                collector = new UserActionCollector(activity, window, manager);
                sCollectors.put(activity, collector);
            }
            return collector;
        }
    }

    public static UserActionCollector unbind(Activity activity) {
        synchronized (sCollectors) {
            UserActionCollector collector = sCollectors.remove(activity);
            if (collector != null) {
                collector.destroy();
            }
            return collector;
        }
    }

    private final Activity mActivity;
    private final Window.Callback mOriginalCallback;
    private final GestureDetector mGestureDetector;
    private final ActionLogManager mActionLogManager;
    private final Window mWindow;

    private UserActionCollector(Activity activity, Window window, ActionLogManager manager) {
        mActivity = activity;
        mWindow = window;
        if (mWindow != null) {
            mOriginalCallback = mWindow.getCallback();
            mWindow.setCallback(this);
        } else {
            mOriginalCallback = null;
        }

        // set actionlog
        mActionLogManager = manager;

        mGestureDetector = new GestureDetector(activity, this);
//        mGestureDetector.setOnDoubleTapListener(this);
    }

    public void destroy() {
        if (mWindow != null) {
            mWindow.setCallback(mOriginalCallback);
        }
    }


    @Override
    public boolean dispatchKeyEvent(KeyEvent event) {
        sLogger.d("dispatchKeyEvent");

        if (mActionLogManager != null) {
            try {
                JSONObject data = buildEventTypedData(
                        mActivity,
                        "key_event",
                        buildKeyEventData(mActivity, event));
                mActionLogManager.push(new ActionLog(ActionLog.Type.USER_ACTION, data));
            } catch (JSONException e) {
                // do nothing
            }
        }

        return mOriginalCallback.dispatchKeyEvent(event);
    }

    @TargetApi(11)
    @Override
    public boolean dispatchKeyShortcutEvent(KeyEvent event) {
        sLogger.d("dispatchKeyShortcutEvent");

        if (mActionLogManager != null) {
            try {
                JSONObject data = buildEventTypedData(
                        mActivity,
                        "key_shortcut_event",
                        buildKeyEventData(mActivity, event));
                mActionLogManager.push(new ActionLog(ActionLog.Type.USER_ACTION, data));
            } catch (JSONException e) {
                // do nothing
            }
        }

        return mOriginalCallback.dispatchKeyShortcutEvent(event);
    }

    @Override
    public boolean dispatchTouchEvent(MotionEvent event) {
        sLogger.d("dispatchTouchEvent");
        mGestureDetector.onTouchEvent(event);
        return mOriginalCallback.dispatchTouchEvent(event);
    }

    @Override
    public boolean dispatchTrackballEvent(MotionEvent event) {
        sLogger.d("dispatchTrackballEvent");

        if (mActionLogManager != null) {

            try {
                JSONObject data = buildEventTypedData(
                        mActivity,
                        "trackball_event",
                        buildMotionEventData(mActivity, event));
                mActionLogManager.push(new ActionLog(ActionLog.Type.USER_ACTION, data));
            } catch (JSONException e) {
                // do nothing
            }
        }

        return mOriginalCallback.dispatchTrackballEvent(event);
    }

    @Override
    public boolean dispatchGenericMotionEvent(MotionEvent event) {

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            // this case gesture detector will feed actionlog

            mGestureDetector.onGenericMotionEvent(event);

        } else if (mActionLogManager != null) {

            try {
                JSONObject data = buildEventTypedData(
                        mActivity,
                        "generic_motion_event",
                        buildMotionEventData(mActivity, event));
                mActionLogManager.push(new ActionLog(ActionLog.Type.USER_ACTION, data));
            } catch (JSONException e) {
                // do nothing
            }
        }

        sLogger.d("dispatchGenericMotionEvent");
        return mOriginalCallback.dispatchGenericMotionEvent(event);
    }

    @Override
    public boolean dispatchPopulateAccessibilityEvent(AccessibilityEvent event) {
        sLogger.d("dispatchPopulateAccessibilityEvent");
        return mOriginalCallback.dispatchPopulateAccessibilityEvent(event);
    }

    @Nullable
    @Override
    public View onCreatePanelView(int featureId) {
        sLogger.d("onCreatePanelView");
        return mOriginalCallback.onCreatePanelView(featureId);
    }

    @Override
    public boolean onCreatePanelMenu(int featureId, Menu menu) {
        sLogger.d("onCreatePanelMenu");
        return mOriginalCallback.onCreatePanelMenu(featureId, menu);
    }

    @Override
    public boolean onPreparePanel(int featureId, View view, Menu menu) {
        sLogger.d("onPreparePanel");
        return mOriginalCallback.onPreparePanel(featureId, view, menu);
    }

    @Override
    public boolean onMenuOpened(int featureId, Menu menu) {
        sLogger.d("onMenuOpened");
        return mOriginalCallback.onMenuOpened(featureId, menu);
    }

    @Override
    public boolean onMenuItemSelected(int featureId, MenuItem item) {
        sLogger.d("onMenuItemSelected");
        return mOriginalCallback.onMenuItemSelected(featureId, item);
    }

    @Override
    public void onWindowAttributesChanged(WindowManager.LayoutParams attrs) {
        sLogger.d("onWindowAttributesChanged");
        mOriginalCallback.onWindowAttributesChanged(attrs);
    }

    @Override
    public void onContentChanged() {
        sLogger.d("onContentChanged");
        mOriginalCallback.onContentChanged();
    }

    @Override
    public void onWindowFocusChanged(boolean hasFocus) {
        sLogger.d("onWindowFocusChanged");
        mOriginalCallback.onWindowFocusChanged(hasFocus);
    }

    @Override
    public void onAttachedToWindow() {
        sLogger.d("onAttachedFromWindow");
        mOriginalCallback.onAttachedToWindow();
    }

    @Override
    public void onDetachedFromWindow() {
        sLogger.d("onDetatchedFromWindow");
        mOriginalCallback.onDetachedFromWindow();
    }

    @Override
    public void onPanelClosed(int featureId, Menu menu) {
        sLogger.d("onPanelClosed");
        mOriginalCallback.onPanelClosed(featureId, menu);
    }

    @Override
    public boolean onSearchRequested() {
        sLogger.d("onSearchRequested");
        return mOriginalCallback.onSearchRequested();
    }

    @TargetApi(23)
    @Override
    public boolean onSearchRequested(SearchEvent searchEvent) {
        sLogger.d("onSearchRequested");
        return mOriginalCallback.onSearchRequested(searchEvent);
    }

    @TargetApi(11)
    @Nullable
    @Override
    public ActionMode onWindowStartingActionMode(ActionMode.Callback callback) {
        sLogger.d("onWindowStartingActionMode");
        return mOriginalCallback.onWindowStartingActionMode(callback);
    }

    @TargetApi(23)
    @Nullable
    @Override
    public ActionMode onWindowStartingActionMode(ActionMode.Callback callback, int type) {
        sLogger.d("onWindowStartingActionMode");
        return mOriginalCallback.onWindowStartingActionMode(callback, type);
    }

    @Override
    public void onActionModeStarted(ActionMode mode) {
        sLogger.d("onActionModeStarted");
        mOriginalCallback.onActionModeStarted(mode);
    }

    @Override
    public void onActionModeFinished(ActionMode mode) {
        sLogger.d("onActionModeFinished");
        mOriginalCallback.onActionModeFinished(mode);
    }


    // Gesture destector OnDoubleTapListener

    @Override
    public boolean onSingleTapConfirmed(MotionEvent event) {
        sLogger.d("onSingleTapConfirmed");

        if (mActionLogManager != null) {
            try {
                JSONObject data = buildEventTypedData(
                        mActivity,
                        "single_tap_confirmed",
                        buildMotionEventData(mActivity, event));
                mActionLogManager.push(new ActionLog(ActionLog.Type.USER_ACTION, data));
            } catch (JSONException e) {
                // do nothing
            }
        }
        return false;
    }

    @Override
    public boolean onDoubleTap(MotionEvent event) {
        sLogger.d("onDoubleTap");

        if (mActionLogManager != null) {
            try {
                JSONObject data = buildEventTypedData(
                        mActivity,
                        "double_tap",
                        buildMotionEventData(mActivity, event));
                mActionLogManager.push(new ActionLog(ActionLog.Type.USER_ACTION, data));
            } catch (JSONException e) {
                // do nothing
            }
        }
        return false;
    }

    @Override
    public boolean onDoubleTapEvent(MotionEvent event) {
        sLogger.d("onDoubleTapEvent");

        if (mActionLogManager != null) {
            try {
                JSONObject data = buildEventTypedData(
                        mActivity,
                        "double_tap_event",
                        buildMotionEventData(mActivity, event));
                mActionLogManager.push(new ActionLog(ActionLog.Type.USER_ACTION, data));
            } catch (JSONException e) {
                // do nothing
            }
        }
        return false;
    }


    // Gesture destector OnGestureListener

    @Override
    public boolean onDown(MotionEvent event) {
        sLogger.d("onDown");

        if (mActionLogManager != null) {
            try {
                JSONObject data = buildEventTypedData(
                        mActivity,
                        "down",
                        buildMotionEventData(mActivity, event));
                mActionLogManager.push(new ActionLog(ActionLog.Type.USER_ACTION, data));
            } catch (JSONException e) {
                // do nothing
            }
        }
        return false;
    }

    @Override
    public void onShowPress(MotionEvent event) {
        sLogger.d("onShowPress");
    }

    @Override
    public boolean onSingleTapUp(MotionEvent event) {
        sLogger.d("onSingleTapUp");

        if (mActionLogManager != null) {
            try {
                JSONObject data = buildEventTypedData(
                        mActivity,
                        "single_tap_up",
                        buildMotionEventData(mActivity, event));
                mActionLogManager.push(new ActionLog(ActionLog.Type.USER_ACTION, data));
            } catch (JSONException e) {
                // do nothing
            }
        }
        return false;
    }

    @Override
    public boolean onScroll(MotionEvent e1, MotionEvent e2, float distanceX, float distanceY) {
        sLogger.d("onScroll");

        if (mActionLogManager != null) {
            try {
                JSONObject data = buildEventTypedData(
                        mActivity,
                        "scroll",
                        buildMotionEventData(mActivity, e1),
                        buildMotionEventData(mActivity, e2));

                data.put("distance_x", distanceX);
                data.put("distance_y", distanceY);

                mActionLogManager.push(new ActionLog(ActionLog.Type.USER_ACTION, data));
            } catch (JSONException e) {
                // do nothing
            }
        }
        return false;
    }

    @Override
    public void onLongPress(MotionEvent event) {
        sLogger.d("onLongPress");

        if (mActionLogManager != null) {
            try {
                JSONObject data = buildEventTypedData(
                        mActivity,
                        "long_press",
                        buildMotionEventData(mActivity, event));

                mActionLogManager.push(new ActionLog(ActionLog.Type.USER_ACTION, data));
            } catch (JSONException e) {
                // do nothing
            }
        }
    }

    @Override
    public boolean onFling(MotionEvent e1, MotionEvent e2, float velocityX, float velocityY) {
        sLogger.d("onFling");

        if (mActionLogManager != null) {
            try {
                JSONObject data = buildEventTypedData(
                        mActivity,
                        "fling",
                        buildMotionEventData(mActivity, e1),
                        buildMotionEventData(mActivity, e2));

                data.put("velocity_x", velocityX);
                data.put("velocity_y", velocityY);

                mActionLogManager.push(new ActionLog(ActionLog.Type.USER_ACTION, data));
            } catch (JSONException e) {
                // do nothing
            }
        }
        return false;
    }

    /**
     * This method is copy of defined in hidden android source code
     * <p/>
     * <p/>
     * Returns a string that represents the symbolic name of the specified unmasked action
     * such as "ACTION_DOWN", "ACTION_POINTER_DOWN(3)" or an equivalent numeric constant
     * such as "UNKNOWN(35)" if unknown.
     *
     * @param event The event.
     * @return The symbolic name of the specified action.
     */
    private static String actionToString(MotionEvent event) {
        int action = event.getAction();
        switch (action) {
            case MotionEvent.ACTION_DOWN:
                return "ACTION_DOWN";
            case MotionEvent.ACTION_UP:
                return "ACTION_UP";
            case MotionEvent.ACTION_CANCEL:
                return "ACTION_CANCEL";
            case MotionEvent.ACTION_OUTSIDE:
                return "ACTION_OUTSIDE";
            case MotionEvent.ACTION_MOVE:
                return "ACTION_MOVE";
            case MotionEvent.ACTION_HOVER_MOVE:
                return "ACTION_HOVER_MOVE";
            case MotionEvent.ACTION_SCROLL:
                return "ACTION_SCROLL";
            case MotionEvent.ACTION_HOVER_ENTER:
                return "ACTION_HOVER_ENTER";
            case MotionEvent.ACTION_HOVER_EXIT:
                return "ACTION_HOVER_EXIT";
            case MotionEvent.ACTION_BUTTON_PRESS:
                return "ACTION_BUTTON_PRESS";
            case MotionEvent.ACTION_BUTTON_RELEASE:
                return "ACTION_BUTTON_RELEASE";
        }
        int index = (action & MotionEvent.ACTION_POINTER_INDEX_MASK) >> MotionEvent.ACTION_POINTER_INDEX_SHIFT;
        switch (action & MotionEvent.ACTION_MASK) {
            case MotionEvent.ACTION_POINTER_DOWN:
                return "ACTION_POINTER_DOWN(" + index + ")";
            case MotionEvent.ACTION_POINTER_UP:
                return "ACTION_POINTER_UP(" + index + ")";
            default:
                return "UNKNOWN(" + Integer.toString(action) + ")";
        }
    }


    /**
     * This method is copy of defined in hidden android source code
     *
     * Returns a string that represents the symbolic name of the specified action
     * such as "ACTION_DOWN", or an equivalent numeric constant such as "UNKNOWN(35)" if unknown.
     *
     * @param event The event.
     * @return The symbolic name of the specified action.
     */
    public static String actionToString(KeyEvent event) {
        int action = event.getAction();
        switch (action) {
            case KeyEvent.ACTION_DOWN:
                return "ACTION_DOWN";
            case KeyEvent.ACTION_UP:
                return "ACTION_UP";
            case KeyEvent.ACTION_MULTIPLE:
                return "ACTION_MULTIPLE";
            default:
                return "UNKNOWN(" + Integer.toString(action) + ")";
        }
    }


    /**
     * This method is copy of defined in hidden android source code
     *
     *
     * Returns a string that represents the symbolic name of the specified combined meta
     * key modifier state flags such as "0", "META_SHIFT_ON",
     * "META_ALT_ON|META_SHIFT_ON" or an equivalent numeric constant such as "0x10000000"
     * if unknown.
     *
     * @param metaState The meta state.
     * @return The symbolic name of the specified combined meta state flags.
     */
    public static String metaStateToString(int metaState) {
        if (metaState == 0) {
            return "0";
        }
        StringBuilder result = null;
        int i = 0;
        while (metaState != 0) {
            final boolean isSet = (metaState & 1) != 0;
            metaState >>>= 1; // unsigned shift!
            if (isSet) {
                final String name = META_SYMBOLIC_NAMES[i];
                if (result == null) {
                    if (metaState == 0) {
                        return name;
                    }
                    result = new StringBuilder(name);
                } else {
                    result.append('|');
                    result.append(name);
                }
            }
            i += 1;
        }
        return result.toString();
    }


    /**
     * This method is copy of defined in hidden android source code
     *
     *
     * Returns a string that represents the symbolic name of the specified combined
     * button state flags such as "0", "BUTTON_PRIMARY",
     * "BUTTON_PRIMARY|BUTTON_SECONDARY" or an equivalent numeric constant such as "0x10000000"
     * if unknown.
     *
     * @param buttonState The button state.
     * @return The symbolic name of the specified combined button state flags.
     */
    public static String buttonStateToString(int buttonState) {
        if (buttonState == 0) {
            return "0";
        }
        StringBuilder result = null;
        int i = 0;
        while (buttonState != 0) {
            final boolean isSet = (buttonState & 1) != 0;
            buttonState >>>= 1; // unsigned shift!
            if (isSet) {
                final String name = BUTTON_SYMBOLIC_NAMES[i];
                if (result == null) {
                    if (buttonState == 0) {
                        return name;
                    }
                    result = new StringBuilder(name);
                } else {
                    result.append('|');
                    result.append(name);
                }
            }
            i += 1;
        }
        return result.toString();
    }

    private static JSONObject buildMotionEventData(Activity activity, MotionEvent event) throws JSONException {
        if (event == null) {
            return null;
        }

        JSONObject data = new JSONObject();
        data.put("action", actionToString(event));

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            data.put("action_button", buttonStateToString(event.getActionButton()));
        }
        data.put("button_state", buttonStateToString(event.getButtonState()));
        data.put("pointer_count", event.getPointerCount());

        JSONArray pointers = new JSONArray();
        for (int i = 0; i < event.getPointerCount(); i++) {
            JSONObject pointer = new JSONObject();
            pointer.put("id", event.getPointerId(i))
                   .put("x", event.getX(i))
                   .put("y", event.getY(i))
                   .put("tool_type", event.getToolType(i));
            pointers.put(pointer);
        }
        data.put("pointers", pointers);

        data.put("meta_state", metaStateToString(event.getMetaState()));
        data.put("history_size", event.getHistorySize());
        data.put("event_time", event.getEventTime());

        // TODO track event source view if possible
//        try {
//            data.put("source", activity.getResources().getResourceName(event.getSource()));
//        } catch (Resources.NotFoundException e) {
//            data.put("source", "unknown(" + event.getSource() + ")");
//        }
        return data;
    }

    private static JSONObject buildKeyEventData(Activity activity, KeyEvent event) throws JSONException {
        if (event == null) {
            return null;
        }

        JSONObject data = new JSONObject();

        data.put("action", actionToString(event));

        String key;
        if (event.getAction() == KeyEvent.ACTION_MULTIPLE &&
                KeyEvent.KEYCODE_UNKNOWN == event.getKeyCode()) {
            key = event.getCharacters();
        } else {
            int keyChar = event.getUnicodeChar();
            if (keyChar == 0) {
                key = "none";
            } else {
                key = String.valueOf((char) keyChar);
            }
        }
        data.put("key", key);

        data.put("meta_state", metaStateToString(event.getMetaState()));
        data.put("repeat_count", event.getRepeatCount());
        data.put("event_time", event.getEventTime());

        // TODO track event source view if possible
//        try {
//            data.put("source", activity.getResources().getResourceName(event.getSource()));
//        } catch (Resources.NotFoundException e) {
//            data.put("source", "unknown(" + event.getSource() + ")");
//        }

        return data;
    }

    private static JSONObject buildEventTypedData(Activity activity, String eventType, JSONObject... eventData) throws
            JSONException {
        JSONObject data = new JSONObject();
        data.put("event_type", eventType);
        data.put("activity", activity.getClass().getName());

        for (int i = 0; i < eventData.length; i++) {
            if (eventData[i] != null) {
                data.put("e" + String.valueOf(i + 1), eventData[i]);
            }
        }
        return data;
    }
}
