package io.solidtech.crash.video;

import android.app.Activity;
import android.app.Application;
import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.Point;
import android.graphics.Rect;
import android.os.Bundle;
import android.util.TypedValue;
import android.view.Display;
import android.view.View;
import android.view.WindowManager;

import java.io.File;
import java.io.FileOutputStream;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.TimeZone;
import java.util.concurrent.atomic.AtomicBoolean;

import io.solidtech.crash.SolidBackgroundService;
import io.solidtech.crash.SolidClient;
import io.solidtech.crash.SolidConstants;
import io.solidtech.crash.SolidThread;
import io.solidtech.crash.utils.DebugUtils;
import io.solidtech.crash.utils.FileUtils;

/**
 * Created by vulpes on 15. 12. 14..
 */
public class Recorder {

    private static final String TAG = Recorder.class.getSimpleName();
    private static final int FILE_QUALITY = 40;
    private static final int WIDTH = 120;
    private static final boolean VERBOSE = true;
    private static final DebugUtils.DebugLogger sLogger = new DebugUtils.DebugLogger(TAG, VERBOSE);

    private static Bitmap sReusableBitmap;

    private static Recorder sRecorder;

    public static synchronized void startMonitor(SolidClient client, Activity activity) {
        sLogger.d("start monitor : " + activity.getClass().getSimpleName());

        if (sRecorder == null) {
            sRecorder = new Recorder(client);
        }
        if (activity != null) {
            sRecorder.startMonitorActivity(activity);
            sRecorder.setEnable();
        }
    }

    public static synchronized void stopMonitor() {
        if (sRecorder != null) {
            sRecorder.destroy();
            sRecorder = null;
        }
    }

    public static synchronized void resumeMonitor() {
        if (sRecorder == null) {
            return;
        }
        sRecorder.setEnable();
    }

    public static synchronized void pauseMonitor() {
        if (sRecorder == null) {
            return;
        }
        sRecorder.setDisable();
    }


    /**
     * Take screenshot of view with background and save as file
     *
     * @param context
     * @param file
     * @param view
     * @param backgroundColor
     * @return file
     */
    private static synchronized File screenCapture(Context context, File file, final View view, int backgroundColor) {
        WindowManager wm = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        Display display = wm.getDefaultDisplay();
        Point size = new Point();
        display.getSize(size);

        // force portrait view
        boolean isPortrait = true;
        if (size.x > size.y) {
            size = new Point(size.y, size.x);
            isPortrait = false;
        }

        if (size.x == 0 || size.y == 0) {
            return null;
        }

        float scaleRatio = (float) WIDTH / size.x;
        int rCanvasWidth = WIDTH;
        int rCanvasHeight = Math.round(size.y * scaleRatio);

        // init reusableBitmap if nonexist, or size changed
        if (sReusableBitmap == null || sReusableBitmap.getWidth() != rCanvasWidth || sReusableBitmap.getHeight() !=
                rCanvasHeight) {
            if (sReusableBitmap != null) {
                sReusableBitmap.recycle();
            }
            sReusableBitmap = Bitmap.createBitmap(rCanvasWidth, rCanvasHeight, Bitmap.Config.RGB_565);
        }


        Canvas canvas = new Canvas(sReusableBitmap);

        // save canvas state
        canvas.save();

        // rotate canvas if app is landscape
        if (!isPortrait) {
            canvas.rotate(-90);
            canvas.translate(-1 * rCanvasHeight, 0);
        }

        // scale canvas
        canvas.scale(scaleRatio, scaleRatio);


        // draw view on center of canvas
        int horizontalMargin = ((isPortrait ? size.x : size.y) - view.getWidth()) / 2;
        int verticalMargin = ((isPortrait ? size.y : size.x) - view.getHeight()) / 2;

        if (horizontalMargin > 0 || verticalMargin > 0) {

            // fill empty background
            sReusableBitmap.eraseColor(0xff000000);

            canvas.translate(horizontalMargin, verticalMargin);

            // fill view background
            Paint paint = new Paint();
            paint.setColor(backgroundColor);
            paint.setStyle(Paint.Style.FILL);
            Rect rect = new Rect(0, 0, view.getWidth(), view.getHeight());
            canvas.drawRect(rect, paint);
        } else {
            // fill view background
            sReusableBitmap.eraseColor(backgroundColor);
        }


        try {
            view.draw(canvas);
            canvas.restore();

            file.createNewFile();
            FileOutputStream ostream = new FileOutputStream(file);
            if (SolidConstants.CAPTURE_PHOTO_TYPE == SolidConstants.ImageType.PNG) {
                sReusableBitmap.compress(Bitmap.CompressFormat.PNG, 100, ostream);
            } else {
                sReusableBitmap.compress(Bitmap.CompressFormat.JPEG, FILE_QUALITY, ostream);
            }
            ostream.close();
        } catch (Exception e) {
//            e.printStackTrace();
        }

        return file;
    }


    private SolidClient mClient;
    private PhotoTaker mTaker;
    private int mInterval;
    private AtomicBoolean mEnabled = new AtomicBoolean(true);

    private Recorder(SolidClient client) {
        mClient = client;
        mInterval = 1000 / client.getConfiguration().videoFrame;
    }

    public synchronized void startMonitorActivity(Activity activity) {
        if (activity == null || activity.isDestroyed() || activity.isFinishing()) {
            sLogger.d("activity is destroyed");
            return;
        }
        if (!FileUtils.isSdcardWritable()) {
            sLogger.e("Sdcard is not writable");
            return;
        }

        // check root view exists
        View rootView = null;
        try {
            rootView = activity.findViewById(android.R.id.content);
        } catch (Throwable t) {
            // view is not ready
            sLogger.d("activity has no rootview");
        }
        if (rootView == null) {
            return;
        }

        if (mTaker != null && mTaker.isAlive()) {
            if (mTaker.getActivity() == activity) {
                return;
            } else {
                mTaker.setActivity(activity, rootView);
            }
            return;
        }
        if (mTaker != null) {
            mTaker.interrupt();
        }
        mTaker = new PhotoTaker(mInterval);
        mTaker.setActivity(activity, rootView);
        mTaker.setDaemon(true);
        mTaker.start();
    }

    public synchronized void destroy() {
        if (mTaker != null) {
            if (mTaker.isAlive()) {
                mTaker.interrupt();
            }
            mTaker = null;
        }
    }

    public void setDisable() {
        synchronized (mEnabled) {
            mEnabled.set(false);
            mEnabled.notify();
        }
    }

    public void setEnable() {
        synchronized (mEnabled) {
            mEnabled.set(true);
            mEnabled.notify();
        }
    }

    private void pushCapturedFile(File file) {
        Application app = mClient.getApplication();
        SolidBackgroundService.remotePushCapturedFile(app, file);
    }

    private class PhotoTaker extends SolidThread {
        private final int mInterval;
        private final SimpleDateFormat mDateFormat;
        private Activity mActivity;
        private View mRootView;
        private int mBackgroundColor;

        public PhotoTaker(int interval) {
            super("SolidPhotoTaker");

            mInterval = interval;
            mDateFormat = new SimpleDateFormat(SolidConstants.DISK_TIME_FORMAT);
            mDateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
        }

        public synchronized void setActivity(Activity activity, View view) {
            mActivity = activity;
            mRootView = view;

            mBackgroundColor = getBackgroundColor(activity);
        }

        public Activity getActivity() {
            return mActivity;
        }

        private int getBackgroundColor(Activity activity) {
//            TypedArray array = mActivity.getTheme().obtainStyledAttributes(new int[]{
//                    android.R.attr.colorBackground
//            });
//            int color = array.getColor(0, 0xFF00FF);
//            array.recycle();
//            return color;

            TypedValue value = new TypedValue();
            activity.getTheme().resolveAttribute(android.R.attr.windowBackground, value, true);
            if (value.type >= TypedValue.TYPE_FIRST_COLOR_INT && value.type <= TypedValue.TYPE_LAST_COLOR_INT) {
                // windowBackground is a color
                return value.data;
            } else {
                // windowBackground is not a color, probably a drawable
                // Drawable d = activity.getResources().getDrawable(value.resourceId);
                return 0xffffff;
            }
        }

        @Override
        public void run() {
            super.run();
            File dir = FileUtils.getCaptureDir(mClient.getApplication(), true);
            if (!dir.exists()) {
                dir.mkdir();
            }

            sLogger.d("start recording ");

            try {
                while (true) {
                    while (!mEnabled.get()) {
                        synchronized (mEnabled) {
                            mEnabled.wait();
                        }
                    }
                    Calendar calendar = new GregorianCalendar();
                    String ext = SolidConstants.CAPTURE_PHOTO_TYPE == SolidConstants.ImageType.JPEG ? ".jpg" : ".png";
                    String filename = SolidConstants.CAPTUREDPHOTO_PREFIX +
                            mDateFormat.format(calendar.getTime()) + ext;

                    File file = new File(dir, filename);

                    Activity activity;
                    View rootView;
                    int backgroundColor;

                    synchronized (this) {
                        activity = mActivity;
                        rootView = mRootView;
                        backgroundColor = mBackgroundColor;
                    }

                    File captureFile = screenCapture(activity, file, rootView, backgroundColor);
                    if (captureFile != null && captureFile.exists()) {
                        pushCapturedFile(captureFile);

                        sLogger.d("captured " + activity.getLocalClassName() + " to " + captureFile.getName());
                    }

                    sleep(mInterval);
                }
            } catch (Throwable e) {
                // do nothing
            }

            sLogger.d("stop recording");
        }
    }
}
