/*
 * Copyright (C) 2016 Dropbeat, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.solidtech.crash.network.implementation.volley;

import android.content.Context;

import com.android.volley.AuthFailureError;
import com.android.volley.NetworkResponse;
import com.android.volley.Request;
import com.android.volley.VolleyError;
import com.android.volley.toolbox.ByteArrayPool;
import com.android.volley.toolbox.HttpStack;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.Map;

import io.solidtech.crash.entities.HttpTransaction;
import io.solidtech.crash.network.HttpParser;
import io.solidtech.crash.network.HttpTracker;
import io.solidtech.crash.utils.HttpInstantCache;

public class BasicNetwork extends com.android.volley.toolbox.BasicNetwork {

    private final Context mContext;

    public BasicNetwork(Context context, HttpStack httpStack) {
        super(httpStack);
        mContext = context;
    }

    public BasicNetwork(Context context, HttpStack httpStack, ByteArrayPool pool) {
        super(httpStack, pool);
        mContext = context;
    }

    @Override
    public NetworkResponse performRequest(Request<?> request) throws VolleyError {
        try {
            NetworkResponse response = super.performRequest(request);
            interceptHttpTransaction(request, response);
            return response;
        } catch (VolleyError e) {
            String reason = e.getMessage();

            if (reason == null) {
                String errName = e.getClass().getName();
                // Volley throws two exceptions without specific error message.
                if (errName.equals("com.android.volley.TimeoutError")
                        || errName.equals("java.lang.RuntimeException")) {
                    reason = errName;
                } else {
                    // Unknown Exception. Putting a string directly is not a good practice.
                    reason = "Unknown";
                }
            }

            // Add error reason to existing transaction.
            HttpTransaction transaction = HttpInstantCache.popTransaction(request.getIdentifier());
            Map<String, String> additionalParams = new HashMap<String, String>();
            additionalParams.put("error", reason);
            transaction.setAdditionalParams(additionalParams);
            HttpInstantCache.setTransaction(request.getIdentifier(), transaction);
            interceptHttpTransaction(request, null);
            throw e;
        }
    }

    /**
     * Intercept HTTP transaction to
     * @param request
     * @param response
     * @throws IOException
     * @throws AuthFailureError
     */
    private void interceptHttpTransaction(Request<?> request, NetworkResponse response) {
        final String DEFAULT_ENCODING = "UTF-8";
        HttpTransaction transaction = HttpInstantCache.popTransaction(request.getIdentifier());
        if (response != null) {
            // Successful HTTP response.
            String responseBody;
            try {
                responseBody = new String(response.data, DEFAULT_ENCODING);
            } catch (UnsupportedEncodingException e) {
                responseBody = "";
            }
            transaction.setRequestBody(responseBody);
            transaction.setBytesReceived(
                    HttpParser.getBytesReceived(response.headers, responseBody.length()));
        }
        HttpTracker.noticeHttpTransaction(mContext, transaction);
    }
}
