/*
 * Copyright (C) 2015 Dropbeat, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.solidtech.crash.network;

import android.content.Context;
import android.util.Log;

import com.squareup.okhttp.MediaType;
import com.squareup.okhttp.OkHttpClient;
import com.squareup.okhttp.Request;
import com.squareup.okhttp.RequestBody;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Map;

import io.solidtech.crash.SolidBackgroundService;
import io.solidtech.crash.SolidClient;
import io.solidtech.crash.SolidConstants;
import io.solidtech.crash.api.common.Defaults;
import io.solidtech.crash.api.common.Resources;
import io.solidtech.crash.entities.HttpPacket;
import io.solidtech.crash.entities.HttpTransaction;
import io.solidtech.crash.exceptions.SolidNetworkException;

public class HttpTracker {

    public static void noticeHttpTransaction(Context context, HttpTransaction transaction) {
        noticeHttpTransaction(
                context,
                transaction.getUrl(), transaction.getHttpMethod(), transaction.getRequestHeaders(),
                transaction.getRequestBody(), transaction.getStartTime(), transaction.getEndTime(),
                transaction.getBytesSent(), transaction.getBytesReceived(), transaction.getResponseCode(),
                transaction.getResponseHeaders(), transaction.getResponseBody(),
                transaction.getAdditionalParams());
    }


    /**
     * Usually used in synchronous networking api because a variable pair such as
     * (`bytesSent`, `bytesReceived`) usually doesn't come together in async api.
     *
     * @param url
     * @param httpMethod
     * @param requestHeaders
     * @param requestBody
     * @param startTime
     * @param endTime
     * @param bytesSent
     * @param bytesReceived
     * @param responseCode
     * @param responseHeaders
     * @param responseBody
     * @param additionalParams
     */
    public static void noticeHttpTransaction(
            Context context, String url, String httpMethod, Map<String, String> requestHeaders,
            String requestBody, Long startTime, Long endTime, Long bytesSent, Long bytesReceived,
            Integer responseCode, Map<String, String> responseHeaders, String responseBody,
            Map<String, String> additionalParams) {
        // Writes to disk.
        writeHttpPacket(context, url, httpMethod, requestHeaders, requestBody, startTime, endTime,
                bytesSent, bytesReceived, responseCode, responseHeaders, responseBody,
                additionalParams);

        try {
            Log.d("HttpTracker", String.format(
                    "Url: %s\nHttpMethod: %s\nRequestHeaders: %s\nRequestBody: %s\nStartTime: %d\nEndTime: %d\nBytesSent: %d\nBytesReceived: %d\nResponseCode: %d\nResponseHeaders: %s\nResponseBody: %s\n",
                    url, httpMethod, requestHeaders.toString(), requestBody, startTime, endTime, bytesSent,
                    bytesReceived, responseCode, responseHeaders.toString(), responseBody));
        } catch (NullPointerException e) {
            // If the app does not receive HTTP response from the server for some reasons,
            // HTTP response variables can be null.
            Log.d("HttpTracker",
                    String.format("Url: %s\nHttpMethod: %s\nRequestHeaders: %s\nRequestBody: %s\nStartTime: %d", url,
                            httpMethod, requestHeaders.toString(), requestBody, startTime));
        }
    }

    /**
     * Writes transaction to disk.
     *
     * @param url
     * @param httpMethod
     * @param requestHeaders
     * @param requestBody
     * @param startTime
     * @param endTime
     * @param bytesSent
     * @param bytesReceived
     * @param responseCode
     * @param responseHeaders
     * @param responseBody
     * @param additionalParams
     */
    private static void writeHttpPacket(
            Context context, String url, String httpMethod, Map<String, String> requestHeaders,
            String requestBody, Long startTime, Long endTime, Long bytesSent, Long bytesReceived,
            Integer responseCode, Map<String, String> responseHeaders, String responseBody,
            Map<String, String> additionalParams) {

        HttpTransaction transaction = new HttpTransaction(
                url, httpMethod, requestHeaders, requestBody, startTime, endTime, bytesSent,
                bytesReceived, responseCode, responseHeaders, responseBody, additionalParams);
        String packetCreatedAt = new SimpleDateFormat(
                SolidConstants.TIME_FORMAT).format(new Date(startTime));
        HttpPacket packet = new HttpPacket(
                transaction, SolidClient.getUser().getIdentifier(), packetCreatedAt);

        // save packet in remote service
        SolidBackgroundService.remotePushHttpPacket(context, packet);
    }
}
