package io.solidtech.crash.network;

import com.squareup.okhttp.Interceptor;
import com.squareup.okhttp.OkHttpClient;
import com.squareup.okhttp.Request;
import com.squareup.okhttp.Response;

import java.io.IOException;
import java.util.concurrent.TimeUnit;

import io.solidtech.crash.SolidClient;
import io.solidtech.crash.api.common.Defaults;
import io.solidtech.crash.environments.ClientInfo;

/**
 * Created by vulpes on 15. 12. 25..
 */
public class HttpClient {

    private static OkHttpClient sClient;
    private static OkHttpClient sFileUploadClient;

    public static OkHttpClient getInstance() {
        if (sClient != null) {
            return sClient;
        }
        synchronized (HttpClient.class) {
            if (sClient != null) {
                return sClient;
            }
            sClient = new OkHttpClient();
            sClient.setConnectTimeout(Defaults.DEFAULT_NETWORK_CONN_TIMEOUT, TimeUnit.MILLISECONDS);
            sClient.setReadTimeout(Defaults.DEFAULT_NETWORK_READ_TIMEOUT, TimeUnit.MILLISECONDS);
            sClient.setWriteTimeout(Defaults.DEFAULT_NETWORK_WRITE_TIMEOUT, TimeUnit.MILLISECONDS);

            sClient.networkInterceptors().add(new UserAgentInterceptor());
            return sClient;
        }
    }

    public static OkHttpClient getFileUploadClient() {
        if (sFileUploadClient != null) {
            return sFileUploadClient;
        }
        synchronized (HttpClient.class) {
            if (sFileUploadClient != null) {
                return sFileUploadClient;
            }
            sFileUploadClient = new OkHttpClient();
            sFileUploadClient.setConnectTimeout(Defaults.DEFAULT_NETWORK_CONN_TIMEOUT, TimeUnit.MILLISECONDS);
            sFileUploadClient.setReadTimeout(Defaults.FILE_UPLOAD_NETWORK_READ_TIMEOUT, TimeUnit.MILLISECONDS);
            sFileUploadClient.setWriteTimeout(Defaults.FILE_UPLOAD_NETWORK_WRITE_TIMEOUT, TimeUnit.MILLISECONDS);

            sFileUploadClient.networkInterceptors().add(new UserAgentInterceptor());
            return sFileUploadClient;
        }
    }

    private static class UserAgentInterceptor implements Interceptor {

        private final String BASIC_USER_AGENT = System.getProperty("http.agent");
        private String mUserAgent;

        @Override
        public Response intercept(Chain chain) throws IOException {
            Request request = chain.request();
            Request newReq = request.newBuilder()
                                    .header("User-Agent", getUserAgent())
                                    .build();
            return chain.proceed(newReq);
        }

        private synchronized String getUserAgent() {
            if (mUserAgent == null) {
                StringBuilder userAgentBuilder = new StringBuilder();
                userAgentBuilder.append(BASIC_USER_AGENT);

                ClientInfo info = SolidClient.getClientInfo();
                if (info != null) {
                    userAgentBuilder.append(" ")
                                    .append(info.getSolidClientName() + "/" + info.getSolidClientVersion())
                                    .append(" ")
                                    .append(info.getAppPackageName() + "/" + info.getAppVersionName())
                                    .append("(" + info.getAppVersionCode() + ")");
                }
                mUserAgent = userAgentBuilder.toString();
            }
            return mUserAgent == null ? BASIC_USER_AGENT : mUserAgent;
        }
    }
}
