package io.solidtech.crash.entities;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.TimeZone;
import java.util.UUID;

import io.solidtech.crash.SolidClient;
import io.solidtech.crash.SolidConstants;
import io.solidtech.crash.SolidUserInfo;
import io.solidtech.crash.actionlog.ActionLog;
import io.solidtech.crash.environments.ActivityStackInfo;
import io.solidtech.crash.environments.ClientInfo;
import io.solidtech.crash.environments.DeviceInfo;
import io.solidtech.crash.environments.ExceptionInfo;
import io.solidtech.crash.environments.GeolocationInfo;
import io.solidtech.crash.environments.NetworkInfo;
import io.solidtech.crash.utils.JsonUtils;

/**
 * Created by vulpes on 16. 1. 23..
 */
public class CrashReport implements Serializable {

    private String uuid;
    private String createdAt;
    private String identifier;
    private String user;
    private String client;
    private String network;
    private String environment;
    private String geolocation;
    private String exceptionInfo;
    private String activityStack;
    private String userCustomData;
    private String actionLogs;

    public static class Builder {
        SolidClient client;
        String identifier;
        GeolocationInfo geolocationInfo;
        SolidUserInfo userInfo;
        ExceptionInfo exceptionInfo;
        ActivityStackInfo activityStackInfo;
        JSONObject userCustomData;
        List<ActionLog> actionLogs;

        public Builder(SolidClient client, String identifier) {
            this.client = client;
            this.identifier = identifier;
            this.actionLogs = new ArrayList<>();
            this.geolocationInfo = GeolocationInfo.getCachedInstance();
        }

        public Builder setException(Thread thread, Throwable e) {
            exceptionInfo = ExceptionInfo.create(client, thread, e);
            return this;
        }

        public Builder appendUser(SolidUserInfo user) {
            userInfo = user;
            return this;
        }

        public Builder appendActivityStackInfo(ActivityStackInfo info) {
            activityStackInfo = info;
            return this;
        }

        public Builder appendUserCustomData(JSONObject data) {
            if (data != null && data.length() > 0) {
                userCustomData = data;
            } else {
                userCustomData = null;
            }
            return this;
        }

        public Builder appendActionLogs(List<ActionLog> actionLogs) {
            if (actionLogs != null) {
                this.actionLogs.addAll(actionLogs);
            }
            return this;
        }

        public CrashReport build() {
            return new CrashReport(
                    client,
                    identifier,
                    geolocationInfo,
                    userInfo,
                    exceptionInfo,
                    activityStackInfo,
                    actionLogs,
                    userCustomData);
        }
    }

    public CrashReport(SolidClient client,
                       String identifier,
                       GeolocationInfo geolocationInfo,
                       SolidUserInfo userInfo,
                       ExceptionInfo exceptionInfo,
                       ActivityStackInfo activityStackInfo,
                       List<ActionLog> actionLogs,
                       JSONObject userCustomData) {

        this.uuid = UUID.randomUUID().toString();
        this.identifier = identifier;
        this.network = robustToString(NetworkInfo.create(client).toJSONObject());
        this.client = robustToString(ClientInfo.create(client).toJSONObject());
        this.environment = robustToString(DeviceInfo.create(client).toJSONObject());
        this.geolocation = robustToString(geolocationInfo != null ? geolocationInfo.toJSONObject() : null);
        this.user = robustToString(userInfo != null ? userInfo.toJSONObject() : null);
        this.exceptionInfo = robustToString(exceptionInfo != null ? exceptionInfo.toJSONArray() : null);
        this.activityStack = robustToString(activityStackInfo != null ? activityStackInfo.toJSONArray() : null);
        this.userCustomData = userCustomData != null ? userCustomData.toString() : null;

        JSONArray actionLogsArray = new JSONArray();
        if (actionLogs != null) {
            for (ActionLog actionLog : actionLogs) {
                actionLogsArray.put(actionLog.toJSONObject());
            }
        }
        this.actionLogs = actionLogsArray.toString();

        GregorianCalendar calendar = new GregorianCalendar();
        SimpleDateFormat format = new SimpleDateFormat(SolidConstants.TIME_FORMAT);
        format.setTimeZone(TimeZone.getTimeZone("UTC"));
        this.createdAt = format.format(calendar.getTimeInMillis());
    }

    public String getUuid() {
        return uuid;
    }

    public String getCreatedAt() {
        return createdAt;
    }

    public JSONObject toJson() {
        try {

            JSONObject data = new JSONObject();
            data.put("client", client != null ? new JSONObject(client) : JSONObject.NULL);
            data.put("network", network != null ? new JSONObject(network) : JSONObject.NULL);
            data.put("environment", environment != null ? new JSONObject(environment) : JSONObject.NULL);
            data.put("user", user != null ? new JSONObject(user) : JSONObject.NULL);
            data.put("geolocation", geolocation != null ? new JSONObject(geolocation) : JSONObject.NULL);
            data.put("activity_stack", activityStack != null ? new JSONArray(activityStack) : new JSONArray());
            data.put("user_custom_data", userCustomData != null ? new JSONObject(userCustomData) : JSONObject.NULL);
            data.put("error", exceptionInfo != null ? new JSONArray(exceptionInfo) : JSONObject.NULL);
            data.put("action_logs", actionLogs != null ? new JSONArray(actionLogs) : JSONObject.NULL);
            data.put("uuid", JsonUtils.wrapNull(uuid));

            JSONObject obj = new JSONObject();
            obj.put("data", data);
            obj.put("user_uid", JsonUtils.wrapNull(identifier));
            obj.put("created_at", JsonUtils.wrapNull(createdAt));

            return obj;
        } catch (JSONException e) {
            e.printStackTrace();
            return null;
        }
    }

    private static String robustToString(Object obj) {
        try {
            return obj.toString();
        } catch (Throwable t) {
            return null;
        }
    }
}
