package io.solidtech.crash;

import android.os.Handler;
import android.os.Looper;

import java.util.concurrent.atomic.AtomicBoolean;

import io.solidtech.crash.exceptions.SolidRuntimeException;
import io.solidtech.crash.utils.DebugUtils;

/**
 * Created by vulpes on 2015. 12. 3..
 */
public class AnrWatchDog extends SolidThread {

    public interface AnrListener {
        void onAnrException(AnrException e);
    }

    private static final String TAG = AnrWatchDog.class.getSimpleName();
    private static final boolean VERBOSE = false;
    private static final DebugUtils.DebugLogger sLogger = new DebugUtils.DebugLogger(TAG, VERBOSE);

    private static AnrWatchDog sInst;

    public static class AnrException extends RuntimeException {
        public AnrException() {
            super("ANR occurred");
        }
    }

    public static synchronized void startWatch(long timeout, AnrListener listener) {
        sLogger.d("start watch");
        if (sInst != null && sInst.isAlive()) {
            // To prevent ANR thread from being started more than once.
            return;
        }
        sInst = new AnrWatchDog(new Handler(Looper.getMainLooper()), listener);
        sInst.start();
        sInst.setTimeout(timeout);
    }

    public static boolean isPaused() {
        if (isRunning()) {
            return !sInst.mEnabled.get();
        }
        return false;
    }

    public static void pauseWatch() {
        if (isRunning()) {
            sInst.mEnabled.set(false);
        }
    }

    public static void resumeWatch() {
        if (isRunning()) {
            sInst.mEnabled.set(true);
            synchronized (sInst.mEnabled) {
                sInst.mEnabled.notify();
            }
        }
    }

    public static synchronized void stopWatch() {
        sLogger.d("stop watch");
        if (isRunning()) {
            sInst.interrupt();
            sInst = null;
        }
    }

    public static synchronized boolean isRunning() {
        return sInst != null && sInst.isAlive();
    }

    private final Handler mMainHandler;
    private final AnrListener mListener;
    private long mTimeout;
    private int mTick;
    private AtomicBoolean mEnabled = new AtomicBoolean(true);

    private AnrWatchDog(Handler handler, AnrListener listener) {
        super("ANR_watchdog");
        mMainHandler = handler;
        mListener = listener;
        mTick = 0;
    }

    public void broadcast(AnrException e) {
        sLogger.d("invoke exception");
        if (mListener != null) {
            mListener.onAnrException(e);
        }
    }

    @Override
    public void run() {
        try {
            while (true) {
                while (!mEnabled.get()) {
                    sLogger.d("paused");
                    synchronized (mEnabled) {
                        mEnabled.wait();
                    }
                }
                final int tick = mTick;
                countTick();
                sleep(mTimeout);
                if (tick == mTick) {
                    sLogger.d("broadcast anr");
                    broadcast(new AnrException());
                    return;
                }
            }
        } catch (InterruptedException e) {
            sLogger.d("Anr timer interrupted");
            // do nothing
        }
    }

    private void setTimeout(long timeout) {
        mTimeout = timeout;
    }

    private void countTick() {
        sLogger.d("tick..");
        mMainHandler.post(new Runnable() {
            @Override
            public void run() {
                sLogger.d("tock..");
                mTick = (mTick + 1) % 10;
            }
        });
    }
}
