package io.solidtech.crash.video;

import android.content.Context;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.TimeZone;
import java.util.concurrent.atomic.AtomicBoolean;

import io.solidtech.crash.SolidClient;
import io.solidtech.crash.SolidConstants;
import io.solidtech.crash.SolidThread;
import io.solidtech.crash.utils.DebugUtils;
import io.solidtech.crash.utils.FileUtils;

/**
 * Created by vulpes on 16. 1. 18..
 */
public class CapturedFileManager {

    private static final int MAX_QUEUE_SIZE = 20 * 1000;
    private static final long EXPIRE_CHECK_INTERVAL = 5 * 60 * 1000;
    private static final long CAPTURED_FILE_EXPIRE_TIME = SolidConstants.VIDEO_EXPIRE_TIME;

    private static final String TAG = CapturedFileManager.class.getSimpleName();
    private static final boolean VERBOSE = false;
    private static final DebugUtils.DebugLogger sLogger = new DebugUtils.DebugLogger(TAG, VERBOSE);

    private static CapturedFileManager sInstance;

    public static synchronized CapturedFileManager getInstance(SolidClient client) {
        if (sInstance == null) {
            sInstance = new CapturedFileManager(client);
        }
        return sInstance;
    }

    private final SolidClient mClient;
    private final SolidThread mExpireThread;
    private final List<CapturedFile> mCapturedFiles = new ArrayList<>();
    private SolidThread mInitializeThread;
    private AtomicBoolean mIsReady = new AtomicBoolean(false);

    private CapturedFileManager(SolidClient client) {
        mClient = client;

        mExpireThread = new SolidThread("SolidCapturedFileExpireThread") {
            @Override
            public void run() {
                super.run();
                try {
                    while (true) {
                        while (!mIsReady.get()) {
                            synchronized (mIsReady) {
                                mIsReady.wait();
                            }
                        }
                        sleep(EXPIRE_CHECK_INTERVAL);
                        try {
                            cleanUp();
                        } catch (IOException e) {
                            // skip
                        }
                    }
                } catch (InterruptedException e) {
                    // do nothing
                }
            }
        };
        mExpireThread.setDaemon(true);
        mExpireThread.start();

        initialize();
    }

    public void pushCapturedFile(File file) {
        synchronized (mCapturedFiles) {
            mCapturedFiles.add(new CapturedFile(file));
        }
        sLogger.d("pushed:" + file.getName());
    }

    /**
     * Get all captured files from manager
     * This function should be called after manager initialize finished.
     *
     * @return files or null (when manager is not initialized)
     */
    public List<File> getFiles() {
        if (!mIsReady.get()) {
            return null;
        }
        List<CapturedFile> copy;
        synchronized (mCapturedFiles) {
            copy = new ArrayList<>(mCapturedFiles);
        }
        List<File> files = new ArrayList<>();

        for (CapturedFile cp : copy) {
            files.add(cp.getFile());
        }
        return files;
    }

    /**
     * Pop all captured files from manager
     * This function should be called after manager initialize finished.
     *
     * @return files or null (when manager is not initialized)
     */
    public List<File> popAll() {
        if (!mIsReady.get()) {
            return null;
        }
        List<CapturedFile> copy;
        synchronized (mCapturedFiles) {
            copy = new ArrayList<>(mCapturedFiles);
            mCapturedFiles.clear();
        }
        List<File> files = new ArrayList<>();

        for (CapturedFile cp : copy) {
            files.add(cp.getFile());
        }
        return files;
    }

    private void initialize() {
        if (mIsReady.get() || (mInitializeThread != null && mInitializeThread.isAlive())) {
            return;
        }
        mInitializeThread = new SolidThread("CapturedFileManagerInitThread") {
            @Override
            public void run() {
                super.run();

                int currentQueueSize = mCapturedFiles.size();
                List<CapturedFile> files = readAndExpireCapturedFiles(
                        mClient.getApplication(),
                        MAX_QUEUE_SIZE - currentQueueSize);
                if (files != null && files.size() > 0) {
                    synchronized (mCapturedFiles) {
                        mCapturedFiles.addAll(0, files);
                    }
                }
                synchronized (mIsReady) {
                    mIsReady.set(true);
                    mIsReady.notifyAll();
                }
                sLogger.d("captured file manager initialized :" + mCapturedFiles.size());
            }
        };
        mInitializeThread.setDaemon(true);
        mInitializeThread.start();
    }

    private void cleanUp() throws IOException, InterruptedException {
        Calendar calendar = GregorianCalendar.getInstance();
        Date now = new Date(calendar.getTimeInMillis());
        List<File> removeFiles = new ArrayList<>();

        synchronized (mCapturedFiles) {
            int totalSize = mCapturedFiles.size();
            for (CapturedFile file : new ArrayList<>(mCapturedFiles)) {
                Date expire = new Date(file.getWhen().getTime() + CAPTURED_FILE_EXPIRE_TIME);
                if (expire.before(now) || totalSize - removeFiles.size() > MAX_QUEUE_SIZE) {
                    CapturedFile capturedFile = mCapturedFiles.remove(0);
                    removeFiles.add(capturedFile.getFile());

                    sLogger.d("clean up " + capturedFile.getFile().getName());
                } else {
                    break;
                }
            }
        }

        for (File removeFile : removeFiles) {
            if (removeFile.exists()) {
                removeFile.delete();
            }
        }
    }

    private List<CapturedFile> readAndExpireCapturedFiles(Context context, int limit) {
        List<CapturedFile> capturedFiles = new ArrayList<>();
        List<File> removeFiles = new ArrayList<>();

        File dir = FileUtils.getCaptureDir(context, true);
        if (dir != null) {
            File[] files = dir.listFiles(new FilenameFilter() {
                @Override
                public boolean accept(File dir, String filename) {
                    String ext = SolidConstants.CAPTURE_PHOTO_TYPE == SolidConstants.ImageType.JPEG ? ".jpg" : ".png";
                    return filename.startsWith(SolidConstants.CAPTUREDPHOTO_PREFIX) &&
                            filename.endsWith(ext);
                }
            });

            if (files == null) {
                return capturedFiles;
            }

            SimpleDateFormat format = new SimpleDateFormat(SolidConstants.DISK_TIME_FORMAT);
            format.setTimeZone(TimeZone.getTimeZone("UTC"));

            Calendar calendar = new GregorianCalendar();
            Date now = calendar.getTime();

            for (File file : files) {
                if (!file.exists()) {
                    continue;
                }
                try {
                    String name = file.getName();
                    int extIdx = name.lastIndexOf(".");
                    String whenStr = name.substring(SolidConstants.CAPTUREDPHOTO_PREFIX.length(), extIdx);
                    Date when = format.parse(whenStr);
                    if (now.getTime() - when.getTime() > CAPTURED_FILE_EXPIRE_TIME) {
                        removeFiles.add(file);
                    } else {
                        capturedFiles.add(new CapturedFile(file, when));
                    }
                } catch (ParseException e) {
                    e.printStackTrace();
                }
            }
        }

        Collections.sort(capturedFiles, new Comparator<CapturedFile>() {
            @Override
            public int compare(CapturedFile lhs, CapturedFile rhs) {
                return lhs.getWhen().getTime() - rhs.getWhen().getTime() > 0 ? 1 : -1;
            }
        });

        if (capturedFiles.size() > limit) {
            List<CapturedFile> overflowFiles = capturedFiles.subList(0, capturedFiles.size() - limit);
            for (CapturedFile file : overflowFiles) {
                removeFiles.add(file.getFile());
            }
            capturedFiles = capturedFiles.subList(capturedFiles.size() - limit, capturedFiles.size());
        }

        for (File file : removeFiles) {
            if (file.exists()) {
                file.delete();
            }
        }

        return capturedFiles;
    }

    public static class CapturedFile {
        private Date mWhen;
        private File mFile;

        public CapturedFile(File file) {
            mFile = file;
            Calendar calendar = GregorianCalendar.getInstance();
            mWhen = calendar.getTime();
        }

        public CapturedFile(File file, Date when) {
            mFile = file;
            mWhen = when;
        }

        public Date getWhen() {
            return mWhen;
        }

        public File getFile() {
            return mFile;
        }
    }
}
