package io.solidtech.crash.utils;

import android.content.Context;
import android.content.SharedPreferences;
import android.provider.Settings.Secure;
import android.telephony.TelephonyManager;
import android.text.TextUtils;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.util.UUID;

import io.solidtech.crash.R;


/**
 * Getting UUID from ANDROID_ID or Random
 *
 * reference : http://stackoverflow.com/questions/5088474/how-can-i-get-the-uuid-of-my-android-phone-in-an-application
 * #answer-5628136
 */
public class DeviceUuid {
    /**
     * Returns a unique UUID for the current android device. As with all UUIDs,
     * this unique ID is "very highly likely" to be unique across all Android
     * devices. Much more so than ANDROID_ID is.
     *
     * The UUID is generated by using ANDROID_ID as the base key if appropriate,
     * falling back on TelephonyManager.getDeviceID() if ANDROID_ID is known to
     * be incorrect, and finally falling back on a random UUID that's persisted
     * to SharedPreferences if getDeviceID() does not return a usable value.
     *
     * In some rare circumstances, this ID may change. In particular, if the
     * device is factory reset a new device ID may be generated. In addition, if
     * a user upgrades their phone from certain buggy implementations of Android
     * 2.2 to a newer, non-buggy version of Android, the device ID may change.
     * Or, if a user uninstalls your app on a device that has neither a proper
     * Android ID nor a Device ID, this ID may change on reinstallation.
     *
     * Note that if the code falls back on using TelephonyManager.getDeviceId(),
     * the resulting ID will NOT change after a factory reset. Something to be
     * aware of.
     *
     * Works around a bug in Android 2.2 for many devices when using ANDROID_ID
     * directly.
     *
     * @return a UUID that may be used to uniquely identify your device for most
     * purposes.
     * http://code.google.com/p/android/issues/detail?id=10603
     */
    public static UUID getUuid(Context context) {
        if (sUuid != null) {
            return sUuid;
        }
        synchronized (DeviceUuid.class) {
            if (sUuid != null) {
                return sUuid;
            }
            String prefFileName = context.getString(R.string.solid_preference_name);
            String deviceUuidKeyName = context.getString(R.string.solid_preference_key__device_uuid);

            final SharedPreferences prefs = context.getSharedPreferences(prefFileName, Context.MODE_PRIVATE);
            String id = prefs.getString(deviceUuidKeyName, null);

            if (FileUtils.isSdcardReadable()) {
                String cachedId = null;
                try {
                    cachedId = readDeviceIdFromFile(context);
                } catch (IOException e) {
                    // do nothing
                }
                if (TextUtils.isEmpty(cachedId)) {
                    if (id != null && FileUtils.isSdcardWritable()) {
                        try {
                            writeDeviceIdToFile(context, id);
                        } catch (IOException e) {
                            // do nothing
                        }
                    }
                } else if (TextUtils.isEmpty(id)) {
                    id = cachedId;
                } else if (!id.equals(cachedId) && FileUtils.isSdcardWritable()) {
                    try {
                        writeDeviceIdToFile(context, id);
                    } catch (IOException e) {
                        // do nothing
                    }
                }
            }

            if (!TextUtils.isEmpty(id)) {
                // Use the ids previously computed and stored in the
                // prefs file
                sUuid = UUID.fromString(id);
                return sUuid;
            }

            final String androidId = Secure.getString(context.getContentResolver(), Secure.ANDROID_ID);

            // Use the Android ID unless it's broken, in which case
            // fallback on deviceId,
            // unless it's not available, then fallback on a random
            // number which we store
            // to a prefs file
            try {
                if (!"9774d56d682e549c".equals(androidId)) {
                    sUuid = UUID.nameUUIDFromBytes(androidId.getBytes("utf8"));
                } else {
                    final String deviceId = ((TelephonyManager) context.getSystemService(
                            Context.TELEPHONY_SERVICE)).getDeviceId();
                    sUuid = TextUtils.isEmpty(deviceId) ?
                            UUID.randomUUID() :
                            UUID.nameUUIDFromBytes(deviceId.getBytes("utf8"));
                }
            } catch (UnsupportedEncodingException e) {
                throw new RuntimeException(e);
            }

            // Write the value out to the prefs file
            final String deviceId = sUuid.toString();
            prefs.edit().putString(deviceUuidKeyName, deviceId).commit();

            if (FileUtils.isSdcardWritable()) {
                try {
                    writeDeviceIdToFile(context, deviceId);
                } catch (IOException e) {
                    // do nothing
                }
            }
            return sUuid;
        }
    }

    private static UUID sUuid;

    private static String readDeviceIdFromFile(Context context) throws IOException {
        File solidDir = FileUtils.getSolidDir(context, false);
        if (solidDir == null) {
            throw new IOException("Failed to find solid dir");
        }
        File cacheFile = new File(solidDir, context.getString(R.string.solid_external_file_key__device_id));
        if (cacheFile.exists()) {
            FileInputStream fis = new FileInputStream(cacheFile);
            BufferedReader reader = new BufferedReader(new InputStreamReader(fis));
            String line = reader.readLine();
            fis.close();
            return line;
        }
        return null;
    }

    private static void writeDeviceIdToFile(Context context, String deviceId) throws IOException {
        File solidDir = FileUtils.getSolidDir(context, true);
        if (solidDir == null) {
            throw new IOException("Failed to find solid dir");
        }
        File cacheFile = new File(solidDir, context.getString(R.string.solid_external_file_key__device_id));
        if (cacheFile.exists()) {
            cacheFile.delete();
        }
        FileOutputStream fos = new FileOutputStream(cacheFile);
        BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(fos));
        writer.write(deviceId);
        writer.flush();
        writer.close();
        fos.close();
    }
}
