package io.solidtech.crash.utils;

import android.app.Activity;
import android.app.Application;
import android.os.Bundle;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Looper;
import android.os.Message;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * Asynchoronous lifecycle tracker
 * This is helper class for handling events without blocking main UI thread
 */
public class AsyncActivityLifecycleTracker implements Application.ActivityLifecycleCallbacks {

    private interface NotifyRunnable {
        void run(Application.ActivityLifecycleCallbacks callback);
    }

    private static final String TAG = AsyncActivityLifecycleTracker.class.getSimpleName();
    private static final boolean VERBOSE = false;
    private static final DebugUtils.DebugLogger sLogger = new DebugUtils.DebugLogger(TAG, VERBOSE);

    private static final int TYPE_ACTIVITY_CREATED = 0;
    private static final int TYPE_ACTIVITY_STARTED = 1;
    private static final int TYPE_ACTIVITY_RESUMED = 2;
    private static final int TYPE_ACTIVITY_PAUSED = 3;
    private static final int TYPE_ACTIVITY_STOPPED = 4;
    private static final int TYPE_ACTIVITY_DESTROYED = 5;
    private static final int TYPE_ACTIVITY_SAVE_INSTANCE_STATE = 6;

    private static AsyncActivityLifecycleTracker sInstance;

    public static synchronized AsyncActivityLifecycleTracker getInstance(Application app) {
        if (sInstance == null) {
            sInstance = new AsyncActivityLifecycleTracker(app);
        }
        return sInstance;
    }

    private static String getReadableActivityState(int type) {
        switch (type) {
            case TYPE_ACTIVITY_CREATED:
                return "activity created";
            case TYPE_ACTIVITY_STARTED:
                return "activity started";
            case TYPE_ACTIVITY_RESUMED:
                return "activity resumed";
            case TYPE_ACTIVITY_PAUSED:
                return "activity paused";
            case TYPE_ACTIVITY_STOPPED:
                return "activity stopped";
            case TYPE_ACTIVITY_DESTROYED:
                return "activity destroyed";
            case TYPE_ACTIVITY_SAVE_INSTANCE_STATE:
                return "activity save instance state";
        }
        return "undefined";
    }

    private Application mApp;
    private final Map<Application.ActivityLifecycleCallbacks, Handler> mObservers;
    private HandlerThread mHandlerThread;
    private LifecycleHandler mHandler;
    private Handler mMainHandler;

    private AsyncActivityLifecycleTracker(Application app) {
        mApp = app;
        mObservers = Collections.synchronizedMap(new HashMap<Application.ActivityLifecycleCallbacks, Handler>());
        mHandlerThread = new HandlerThread("AsyncActivityLifecycleHandlerThread");
        mHandlerThread.start();

        mHandler = new LifecycleHandler(mHandlerThread.getLooper());
        mMainHandler = new Handler(Looper.getMainLooper());

        mApp.registerActivityLifecycleCallbacks(this);
    }

    public void addObserver(Application.ActivityLifecycleCallbacks callback, Handler handler) {
        mObservers.put(callback, handler);
    }

    public void removeObserver(Application.ActivityLifecycleCallbacks callbacks) {
        mObservers.remove(callbacks);
    }

    /**
     * This method should be called on application's finish code.
     * Because interrupting and destroying thread when app is become finished is optional,
     * (Android kernel will terminate child threads automatically)
     * calling this method also optional
     */
    public void destroy() {
        mApp.unregisterActivityLifecycleCallbacks(this);
        mHandlerThread.interrupt();
    }

    private void notifyCallback(final NotifyRunnable callback) {
        for (Application.ActivityLifecycleCallbacks key : mObservers.keySet()) {

            final Application.ActivityLifecycleCallbacks observer = key;
            Handler handler = mObservers.get(key);

            if (handler != null) {
                handler.post(new Runnable() {
                    @Override
                    public void run() {
                        callback.run(observer);
                    }
                });
            }
        }
    }

    private void notifyOnActivityCreated(final Activity activity, final Bundle savedInstanceState) {
        notifyCallback(new NotifyRunnable() {
            @Override
            public void run(Application.ActivityLifecycleCallbacks callbacks) {
                callbacks.onActivityCreated(activity, savedInstanceState);
            }
        });
    }

    private void notifyOnActivityStarted(final Activity activity) {
        notifyCallback(new NotifyRunnable() {
            @Override
            public void run(Application.ActivityLifecycleCallbacks callbacks) {
                callbacks.onActivityStarted(activity);
            }
        });
    }

    private void notifyOnActivityResumed(final Activity activity) {
        notifyCallback(new NotifyRunnable() {
            @Override
            public void run(Application.ActivityLifecycleCallbacks callbacks) {
                callbacks.onActivityResumed(activity);
            }
        });
    }

    private void notifyOnActivityPaused(final Activity activity) {
        notifyCallback(new NotifyRunnable() {
            @Override
            public void run(Application.ActivityLifecycleCallbacks callbacks) {
                callbacks.onActivityPaused(activity);
            }
        });
    }

    private void notifyOnActivityStopped(final Activity activity) {
        notifyCallback(new NotifyRunnable() {
            @Override
            public void run(Application.ActivityLifecycleCallbacks callbacks) {
                callbacks.onActivityStopped(activity);
            }
        });
    }

    private void notifyOnActivityDestroyed(final Activity activity) {
        notifyCallback(new NotifyRunnable() {
            @Override
            public void run(Application.ActivityLifecycleCallbacks callbacks) {
                callbacks.onActivityDestroyed(activity);
            }
        });
    }

    private void notifyOnActivitySaveInstanceState(final Activity activity, final Bundle outState) {
        notifyCallback(new NotifyRunnable() {
            @Override
            public void run(Application.ActivityLifecycleCallbacks callbacks) {
                callbacks.onActivitySaveInstanceState(activity, outState);
            }
        });
    }

    @Override
    public void onActivityCreated(Activity activity, Bundle savedInstanceState) {
        Message message = new Message();
        message.what = TYPE_ACTIVITY_CREATED;
        message.obj = activity;
        message.setData(savedInstanceState);

        mHandler.handleMessage(message);
    }

    @Override
    public void onActivityStarted(Activity activity) {
        Message message = new Message();
        message.what = TYPE_ACTIVITY_STARTED;
        message.obj = activity;

        mHandler.handleMessage(message);
    }

    @Override
    public void onActivityResumed(Activity activity) {
        Message message = new Message();
        message.what = TYPE_ACTIVITY_RESUMED;
        message.obj = activity;

        mHandler.handleMessage(message);
    }

    @Override
    public void onActivityPaused(Activity activity) {
        Message message = new Message();
        message.what = TYPE_ACTIVITY_PAUSED;
        message.obj = activity;

        mHandler.handleMessage(message);
    }

    @Override
    public void onActivityStopped(Activity activity) {
        Message message = new Message();
        message.what = TYPE_ACTIVITY_STOPPED;
        message.obj = activity;

        mHandler.handleMessage(message);
    }

    @Override
    public void onActivitySaveInstanceState(Activity activity, Bundle outState) {
        Message message = new Message();
        message.what = TYPE_ACTIVITY_SAVE_INSTANCE_STATE;
        message.obj = activity;
        message.setData(outState);

        mHandler.handleMessage(message);
    }

    @Override
    public void onActivityDestroyed(Activity activity) {
        Message message = new Message();
        message.what = TYPE_ACTIVITY_DESTROYED;
        message.obj = activity;

        mHandler.handleMessage(message);
    }

    private class LifecycleHandler extends Handler {

        public LifecycleHandler(Looper looper) {
            super(looper);
        }

        @Override
        public void handleMessage(Message msg) {
            Activity activity = (Activity) msg.obj;
            Bundle bundle = msg.getData();
            int type = msg.what;

            sLogger.d("handle activity state:" + getReadableActivityState(type));

            switch (type) {
                case TYPE_ACTIVITY_CREATED:
                    notifyOnActivityCreated(activity, bundle);
                    break;
                case TYPE_ACTIVITY_STARTED:
                    notifyOnActivityStarted(activity);
                    break;
                case TYPE_ACTIVITY_PAUSED:
                    notifyOnActivityPaused(activity);
                    break;
                case TYPE_ACTIVITY_RESUMED:
                    notifyOnActivityResumed(activity);
                    break;
                case TYPE_ACTIVITY_STOPPED:
                    notifyOnActivityStopped(activity);
                    break;
                case TYPE_ACTIVITY_DESTROYED:
                    notifyOnActivityDestroyed(activity);
                    break;
                case TYPE_ACTIVITY_SAVE_INSTANCE_STATE:
                    notifyOnActivitySaveInstanceState(activity, bundle);
                    break;
                default:
                    return;
            }
        }
    }
}
