/*
 * Copyright (C) 2016 Dropbeat, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.solidtech.crash.network.implementation.volley;

import com.android.volley.AuthFailureError;
import com.android.volley.Request;

import org.apache.http.HttpResponse;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import io.solidtech.crash.entities.HttpTransaction;
import io.solidtech.crash.network.HttpMethod;
import io.solidtech.crash.network.HttpParser;
import io.solidtech.crash.utils.HttpInstantCache;

public class HurlStack extends com.android.volley.toolbox.HurlStack {
    @Override
    public HttpResponse performRequest(
            Request<?> request, Map<String, String> additionalHeaders)
            throws IOException, AuthFailureError {
        // 1. Because Volley has a bug in `BasicNetwork.performRequest`,
        // non-success HTTP response like 400 does not be throwed properly.
        // So, there is no way to intercept every HTTP response without hooking un-processed
        // response here.
        // 2. As response entity is consumed in `volley.toolkit.BasicNetwork`, we can't read responseBody here.
        // So after temporary saving HttpTransaction on memory first, we intercept response body
        // from volley network layer and record it by calling `HttpTracker.noticeHttpTransaction`.
        String url = request.getUrl();

        String httpMethod = HttpMethod.volleyMethodName(request.getMethod());
        Map<String, String> requestHeaders = new HashMap<String, String>();
        requestHeaders.putAll(request.getHeaders());
        requestHeaders.putAll(additionalHeaders);

        String requestBody = "";
        if (request.getBody() != null) {
            // Is it safe to force UTF-8 here?
            requestBody = new String(request.getBody(), "UTF-8");
        }
        long bytesSent = HttpParser.getBytesSent(
                httpMethod, url, request.getHeaders(), requestBody);
        long startTime = System.currentTimeMillis();

        HttpTransaction transaction = new HttpTransaction(
                url, httpMethod, requestHeaders, requestBody, startTime,
                null, bytesSent, null, null, null, null, null);

        // Because following `performRequest` can throw exception.
        // To make sure that HttpTransaction is saved even if there is an error,
        // we should call setTransaction here first.
        HttpInstantCache.setTransaction(request.getIdentifier(), transaction);

        // Original volley contains only following line.
        HttpResponse response = super.performRequest(request, additionalHeaders);

        long endTime = System.currentTimeMillis();
        int responseCode = response.getStatusLine().getStatusCode();
        Map<String, String> responseHeaders =
                HttpParser.httpHeaderToMap(response.getAllHeaders());
        // Undetermined.
        String responseBody = "";
        long bytesReceived = 0;

        transaction.setEndTime(endTime);
        transaction.setBytesReceived(bytesReceived);
        transaction.setResponseCode(responseCode);
        transaction.setResponseHeaders(responseHeaders);
        transaction.setResponseBody(responseBody);

        HttpInstantCache.setTransaction(request.getIdentifier(), transaction);
        return response;
    }
}
