package io.solidtech.crash.actionlog;

import android.util.Log;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;

import io.solidtech.crash.SolidConstants;
import io.solidtech.crash.SolidThread;
import io.solidtech.crash.utils.DebugUtils;

/**
 * Automatically being expired actionlog storage
 *
 * Created by vulpes on 15. 12. 18..
 */
public class ActionLogManager {

    private static final int ACTION_LOG_EXPIRE_TIME = 30 * 1000;
    private static final int ACTION_LOG_CLEANUP_INTERVAL = 1000;

    private static final String TAG = ActionLogManager.class.getSimpleName();
    private static final boolean VERBOSE = false;
    private static final DebugUtils.DebugLogger sLogger = new DebugUtils.DebugLogger(TAG, VERBOSE);

    private final List<ActionLog> mActionLogs = new ArrayList<>();
    private SolidThread mExpireThread;

    public ActionLogManager() {

        mExpireThread = new SolidThread("SolidActionLogExpireThread") {
            @Override
            public void run() {
                super.run();
                try {
                    while (true) {
                        sleep(ACTION_LOG_CLEANUP_INTERVAL);
                        cleanUp();
                    }
                } catch (InterruptedException e) {
                    // do nothing
                }
            }
        };
        mExpireThread.setDaemon(true);
        mExpireThread.start();
    }

    public void destroy() {
        if (mExpireThread != null && mExpireThread.isAlive()) {
            mExpireThread.interrupt();
        }
    }

    public List<ActionLog> getLogs() {
        return new ArrayList<>(mActionLogs);
    }

    public synchronized void push(ActionLog log) {
        if (log.getTime() == null) {
            Calendar calendar = GregorianCalendar.getInstance();
            log.setTime(calendar.getTime());
        }
        for (int i = mActionLogs.size() - 1; i >= 0; i--) {
            ActionLog actionLog = mActionLogs.get(i);
            Date when = actionLog.getTime();
            if (log.getTime().before(when)) {
                mActionLogs.add(i, log);

                sLogger.d("ActionLog<" + log.getType().name() + "> pushed at " + i);
                return;
            }
        }
        mActionLogs.add(log);

        sLogger.d("ActionLog<" + log.getType().name() + "> pushed at " + (mActionLogs.size() - 1));
    }

    private synchronized void cleanUp() {
        Calendar calendar = GregorianCalendar.getInstance();
        Date now = new Date(calendar.getTimeInMillis());
        SimpleDateFormat format = new SimpleDateFormat(SolidConstants.TIME_FORMAT);

        List<ActionLog> copiedList = new ArrayList<>(mActionLogs);

        for (int i = 0; i < copiedList.size(); i++) {
            ActionLog actionLog = copiedList.get(i);
            Date when = actionLog.getTime();
            Date expireTime = new Date(when.getTime() + ACTION_LOG_EXPIRE_TIME);
            if (expireTime.before(now)) {
                mActionLogs.remove(0);

                sLogger.d("ActionLog<" + actionLog.getType().name() + "> at " + format.format(when) +
                        " expired");
                continue;
            }
            break;
        }
    }

    private synchronized void clear() {
        mActionLogs.clear();
    }
}
