package io.solidtech.crash.actionlog;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;

import io.solidtech.crash.SolidClient;
import io.solidtech.crash.SolidThread;
import io.solidtech.crash.environments.ActivityStackInfo;
import io.solidtech.crash.environments.DeviceInfo;
import io.solidtech.crash.environments.NetworkInfo;
import io.solidtech.crash.utils.DebugUtils;
import io.solidtech.crash.utils.JsonUtils;

/**
 * Created by vulpes on 15. 12. 18..
 */
public class ActionLogFactory {

    private static final String TAG = ActionLogFactory.class.getSimpleName();
    private static final boolean VERBOSE = false;
    private static final DebugUtils.DebugLogger sLogger = new DebugUtils.DebugLogger(TAG, VERBOSE);

    private final SolidClient mClient;
    private MonitorThread mMonitorThread;
    private NetworkInfo mLastNetworkInfo;
    private DeviceInfo mLastDeviceInfo;
    private ActivityStackInfo mLastActivityStackInfo;

    public ActionLogFactory(SolidClient client) {
        mClient = client;
        mLastNetworkInfo = NetworkInfo.create(mClient);
        mLastDeviceInfo = DeviceInfo.create(mClient);
        mLastActivityStackInfo = null;
    }

    public void reset() {
        mLastNetworkInfo = NetworkInfo.create(mClient);
        mLastDeviceInfo = DeviceInfo.create(mClient);
        mLastActivityStackInfo = null;
    }

    public synchronized void startMonitor(ActionLogManager manager) {
        stopMonitor();
        mMonitorThread = new MonitorThread(manager);
        mMonitorThread.setDaemon(true);
        mMonitorThread.start();
    }

    public synchronized void stopMonitor() {
        if (mMonitorThread != null && mMonitorThread.isAlive()) {
            mMonitorThread.interrupt();
        }
        mMonitorThread = null;
    }

    public void feedNetworkActionLog(final ActionLogManager manager) {
        ActionLog actionLog = createNetworkActionLog();
        if (actionLog != null) {
            manager.push(actionLog);
        }
    }

    public ActionLog createNetworkActionLog() {
        NetworkInfo lastNetworkInfo;
        NetworkInfo newNetworkInfo;
        synchronized (this) {
            lastNetworkInfo = mLastNetworkInfo;
            newNetworkInfo = NetworkInfo.create(mClient);
            mLastNetworkInfo = newNetworkInfo;
        }
        if (lastNetworkInfo == null) {
            return null;
        }
        try {
            JSONObject networkDiff = JsonUtils.diff(lastNetworkInfo.toJSONObject(), newNetworkInfo.toJSONObject());
            if (networkDiff == null) {
                sLogger.d("network no diff");
                return null;
            }
            sLogger.d(networkDiff.toString());

            Calendar calendar = GregorianCalendar.getInstance();
            return new ActionLog(ActionLog.Type.NETWORK_DIFF, new Date(calendar.getTimeInMillis()), networkDiff);
        } catch (JSONException e) {
            sLogger.d("failed to diff json object");
        }
        return null;
    }

    public void feedEnvActionLog(final ActionLogManager manager) {
        ActionLog actionLog = createEnvActionLog();
        if (actionLog != null) {
            manager.push(actionLog);
        }
    }

    public ActionLog createEnvActionLog() {
        DeviceInfo lastDeviceInfo;
        DeviceInfo newDeviceInfo;
        synchronized (this) {
            lastDeviceInfo = mLastDeviceInfo;
            newDeviceInfo = DeviceInfo.create(mClient);
            mLastDeviceInfo = newDeviceInfo;
        }
        if (lastDeviceInfo == null) {
            return null;
        }
        try {
            JSONObject deviceDiff = JsonUtils.diff(lastDeviceInfo.toJSONObject(), newDeviceInfo.toJSONObject());

            // ignore ram size diff
            if (deviceDiff != null && deviceDiff.optString("free_ram_size") != null && deviceDiff.length() == 1) {
                deviceDiff = null;
            }

            if (deviceDiff == null) {
                sLogger.d("env no diff");
                return null;
            }
            sLogger.d(deviceDiff.toString());

            Calendar calendar = GregorianCalendar.getInstance();
            return new ActionLog(ActionLog.Type.ENVIRONMENT_DIFF, new Date(calendar.getTimeInMillis()), deviceDiff);
        } catch (JSONException e) {
            sLogger.d("failed to diff json object");
        }
        return null;
    }

    public void feedActivityActionLog(final ActionLogManager manager, final ActivityStackInfo info) {
        ActionLog actionLog = createActivityActionLog(info);
        if (actionLog != null) {
            manager.push(actionLog);
        }
    }

    public ActionLog createActivityActionLog(ActivityStackInfo info) {
        JSONObject newStackInfo = new JSONObject();
        JSONObject oldStackInfo = new JSONObject();
        if (info != null) {
            try {
                JSONArray array = info.toJSONArray();
                if (array != null) {
                    newStackInfo.put("activities", array);
                }
            } catch (JSONException e) {
                // do nothing
            }
        }

        if (mLastActivityStackInfo != null) {
            try {
                JSONArray array = mLastActivityStackInfo.toJSONArray();
                if (array != null) {
                    oldStackInfo.put("activities", array);
                }
            } catch (JSONException e) {
                // do nothing
            }
        }

        try {
            JSONObject diff = JsonUtils.diff(oldStackInfo, newStackInfo);

            if (diff == null) {
                sLogger.d("activities no diff");
                return null;
            }
            sLogger.d(diff.toString());
            mLastActivityStackInfo = info;
            Calendar calendar = GregorianCalendar.getInstance();
            return new ActionLog(ActionLog.Type.ACTIVITY_DIFF, new Date(calendar.getTimeInMillis()), diff);
        } catch (JSONException e) {
            sLogger.d("failed to diff json object");
        }
        return null;
    }

    private class MonitorThread extends SolidThread {

        private ActionLogManager mManager;

        public MonitorThread(ActionLogManager manager) {
            super("ActionLogMonitorThread");
            mManager = manager;
        }

        @Override
        public void run() {
            super.run();
            try {
                while (true) {
                    sleep(mClient.getConfiguration().actionLogWatchInterval);
                    feedNetworkActionLog(mManager);
                    feedEnvActionLog(mManager);
                }
            } catch (InterruptedException e) {
                // do nothing
            }
        }
    }
}
