package io.solidtech.crash;

import android.content.Intent;

/**
 * Created by vulpes on 2015. 11. 30..
 */
public class SolidConfiguration {
    public static final SolidConfiguration DEFAUlT = new Builder().build();

    public static final boolean DEFAULT_USE_VIDEO = true;
    public static final int DEFAULT_VIDEO_FRAME = 2;
    public static final boolean DEFAULT_UPLOAD_VIDEO_ONLY_WIFI = false;

    public static final boolean DEFAULT_USE_ANR = true;
    public static final int DEFAULT_ANR_TIMEOUT = 5000;

    public static final boolean DEFAULT_USE_NETWORK_STATE_MONITOR = false;
    public static final boolean DEFAULT_USE_DETAIL_NETWORK_STATE_MONITOR = false;

    public static final boolean DEFAULT_USE_ACTION_LOG = true;
    public static final int DEFAULT_ACTION_LOG_WATCH_INTERVAL = 3000;

    public static class Builder {

        private boolean useVideo;
        private int videoFrame;
        private boolean uploadVideoOnlyWifi;

        private boolean useAnr;
        private int anrTimeout;

        private boolean useNetworkStateMonitor;
        private boolean useDetailNetworkStateMonitor;

        private int actionLogWatchInterval;
        private boolean useActionLog;

        private Intent customCrashNotifyIntent;

        public Builder() {
            useVideo = DEFAULT_USE_VIDEO;
            videoFrame = DEFAULT_VIDEO_FRAME;
            uploadVideoOnlyWifi = DEFAULT_UPLOAD_VIDEO_ONLY_WIFI;

            useAnr = DEFAULT_USE_ANR;
            anrTimeout = DEFAULT_ANR_TIMEOUT;

            useNetworkStateMonitor = DEFAULT_USE_NETWORK_STATE_MONITOR;
            useDetailNetworkStateMonitor = DEFAULT_USE_DETAIL_NETWORK_STATE_MONITOR;

            actionLogWatchInterval = DEFAULT_ACTION_LOG_WATCH_INTERVAL;
            useActionLog = DEFAULT_USE_ACTION_LOG;

            customCrashNotifyIntent = null;
        }

        /**
         * Set crash report video use
         * Default: true
         *
         * @param use
         * @return builder
         */
        public Builder sendVideoOnCrash(boolean use) {
            useVideo = use;
            return this;
        }

        /**
         * Set video upload only when it is connected with wifi
         *
         * @param onlyWifi
         * @return builder
         */
        public Builder uploadVideoOnlyWifi(boolean onlyWifi) {
            uploadVideoOnlyWifi = onlyWifi;
            return this;
        }

        /**
         * Send crash report for ANR
         * Default: true
         *
         * @param use
         * @return builder
         */
        public Builder useAnrCheck(boolean use) {
            useAnr = use;
            return this;
        }

        /**
         * Set ANR check timeout
         * Default: 5000 msec
         *
         * @param msec
         * @return builder
         */
        public Builder setAnrTimeout(int msec) {
            anrTimeout = msec;
            return this;
        }

        /**
         * Track simple network state (wifi connection, mobile network connection)
         * Default: false
         *
         * @param use
         * @return builder
         */
        public Builder useSimpleNetworkStateMonitor(boolean use) {
            useNetworkStateMonitor = true;
            useDetailNetworkStateMonitor = !use;
            return this;
        }

        /**
         * Track detail network state which include wifi name, connection state, etc.
         * Default: false
         *
         * @param use
         * @return builder
         */
        public Builder useDetailNetworkStateMonitor(boolean use) {
            useNetworkStateMonitor = true;
            useDetailNetworkStateMonitor = use;
            return this;
        }

        /**
         * Set custom crash activity intent
         * This intent will be shown when crash occur
         *
         * @param intent
         * @return builder
         */
        public Builder showCustomCrashActivity(Intent intent) {
            customCrashNotifyIntent = intent;
            return this;
        }


        /**
         * Set use action log
         * Default: true
         *
         * @param use
         * @return builder
         */
        public Builder useActionLog(boolean use) {
            useActionLog = use;
            return this;
        }

        /**
         * Set action watch interval
         * Default: 3000
         *
         * @param interval
         * @return builder
         */
        public Builder setActionLogWatchInterval(int interval) {
            actionLogWatchInterval = interval;
            return this;
        }

        public SolidConfiguration build() {
            return new SolidConfiguration(Builder.this);
        }
    }

    public final boolean useVideo;
    public final int videoFrame;
    public final boolean uploadVideoOnlyWifi;

    public final boolean useAnr;
    public final int anrTimeout;

    public final boolean useNetworkStateMonitor;
    public final boolean useDetailNetworkStateMonitor;

    public final int actionLogWatchInterval;
    public final boolean useActionLog;

    public final Intent customCrashNotifyIntent;

    private SolidConfiguration(Builder builder) {
        useVideo = builder.useVideo;
        videoFrame = builder.videoFrame;
        uploadVideoOnlyWifi = builder.uploadVideoOnlyWifi;

        useAnr = builder.useAnr;
        anrTimeout = builder.anrTimeout;

        useNetworkStateMonitor = builder.useNetworkStateMonitor;
        useDetailNetworkStateMonitor = builder.useDetailNetworkStateMonitor;

        customCrashNotifyIntent = builder.customCrashNotifyIntent;

        actionLogWatchInterval = builder.actionLogWatchInterval;
        useActionLog = builder.useActionLog;
    }
}
