package io.solidtech.crash;

import android.app.ActivityManager;
import android.app.Service;
import android.content.Context;
import android.content.Intent;
import android.os.IBinder;
import android.support.annotation.Nullable;
import android.text.TextUtils;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import io.solidtech.crash.entities.CrashReport;
import io.solidtech.crash.entities.HttpPacket;
import io.solidtech.crash.entities.sugar.TrackingInfo;
import io.solidtech.crash.entities.sugar.VideoMaterial;
import io.solidtech.crash.environments.ConnectivityChangeNotifier;
import io.solidtech.crash.exceptions.SolidRuntimeException;
import io.solidtech.crash.network.TrackingInfoSender;
import io.solidtech.crash.utils.DebugUtils;
import io.solidtech.crash.utils.FileUtils;
import io.solidtech.crash.video.CapturedFileManager;
import io.solidtech.crash.video.VideoGenerator;

/**
 * Created by vulpes on 16. 1. 18..
 */
public class SolidBackgroundService extends Service
        implements VideoGenerator.OnVideoGeneratedListener, ConnectivityChangeNotifier.SimpleObserver,
                   TrackingInfoSender.OnCrashSentListener {

    private static final String TAG = SolidBackgroundService.class.getSimpleName();
    private static final boolean VERBOSE = true;
    private static final DebugUtils.DebugLogger sLogger = new DebugUtils.DebugLogger(TAG, VERBOSE);

    private static final String ACTION_PUSH_CRASH = "io.solidtech.crash.action_push_crash";
    private static final String ACTION_PUSH_HTTP_PACKET = "io.solidtech.crash.action_push_http_transaction";
    private static final String ACTION_CREATE_VIDEO = "io.solidtech.crash.action_create_video";
    private static final String ACTION_APP_BECOME_FOREGROUND = "io.solidtech.crash.action_foreground";
    private static final String ACTION_APP_BECOME_BACKGROUND = "io.solidtech.crash.action_background";
    private static final String ACTION_PUSH_CAPTURED_FILE = "io.solidtech.crash.action_push_captured_file";

    private static final String PARAM_CRASH = "crash";
    private static final String PARAM_CAPTURED_FILE = "captured_file";
    private static final String PARAM_HTTP_PACKET = "http_packet";

    public static void remoteFlushAll(Context context) {
        context.startService(new Intent(context, SolidBackgroundService.class));
    }

    public static void remotePushCrash(Context context, CrashReport crash) {
        Intent intent = new Intent(context, SolidBackgroundService.class);
        intent.setAction(ACTION_PUSH_CRASH);
        intent.putExtra(PARAM_CRASH, crash);
        context.startService(intent);
    }

    public static void remoteVideoCreate(Context context) {
        Intent intent = new Intent(context, SolidBackgroundService.class);
        intent.setAction(ACTION_CREATE_VIDEO);
        context.startService(intent);
    }

    public static void remoteAppForegroundNotify(Context context) {
        Intent intent = new Intent(context, SolidBackgroundService.class);
        intent.setAction(ACTION_APP_BECOME_FOREGROUND);
        context.startService(intent);
    }

    public static void remoteAppBackgroundNotify(Context context) {
        Intent intent = new Intent(context, SolidBackgroundService.class);
        intent.setAction(ACTION_APP_BECOME_BACKGROUND);
        context.startService(intent);
    }

    public static void remotePushCapturedFile(Context context, File file) {
        Intent intent = new Intent(context, SolidBackgroundService.class);
        intent.setAction(ACTION_PUSH_CAPTURED_FILE);
        intent.putExtra(PARAM_CAPTURED_FILE, file);
        context.startService(intent);
    }

    public static void remotePushHttpPacket(Context context, HttpPacket packet) {
        Intent intent = new Intent(context, SolidBackgroundService.class);
        intent.setAction(ACTION_PUSH_HTTP_PACKET);
        intent.putExtra(PARAM_HTTP_PACKET, packet);
        context.startService(intent);
    }

    public static boolean isBackgroundProcess(Context context) {
        Context appContext = context.getApplicationContext();

        long myPid = (long) android.os.Process.myPid();
        List<ActivityManager.RunningAppProcessInfo> runningAppProcesses;
        runningAppProcesses = ((ActivityManager) appContext.getSystemService(Context.ACTIVITY_SERVICE))
                .getRunningAppProcesses();

        if (runningAppProcesses != null) {
            for (ActivityManager.RunningAppProcessInfo runningAppProcessInfo : runningAppProcesses) {
                long pid = (long) runningAppProcessInfo.pid;
                String packageName = appContext.getPackageName();
                String processName = packageName + ":solidBackgroundService";

                if (pid == myPid && processName.equals(runningAppProcessInfo.processName)) {
                    return true;
                }
            }
        }
        return false;
    }

    private SolidClient mClient;
    private VideoGenerator mVideoGenerator;
    private CapturedFileManager mFileManager;
    private ConnectivityChangeNotifier mConnectivityNotifier;
    private TrackingInfoSender mInfoSender;

    @Override
    public void onCreate() {
        super.onCreate();
        if (mClient == null) {
            mClient = SolidClient.getInstance();
        }
        if (mClient == null) {
            stopSelf();
            sLogger.d("BackgroundService stopped");
            return;
        }

        mInfoSender = TrackingInfoSender.getInstance(mClient);
        mInfoSender.addOnCrashSentListener(this);

        mVideoGenerator = new VideoGenerator(mClient, this);
        mVideoGenerator.start();

        mFileManager = CapturedFileManager.getInstance(mClient);
        mConnectivityNotifier = ConnectivityChangeNotifier.getInstance(this);
        mConnectivityNotifier.addSimpleObserver(this);
    }

    @Nullable
    @Override
    public IBinder onBind(Intent intent) {
        throw new SolidRuntimeException("Solid video pusher should be started with \'startService\'");
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {

        String action = intent != null ? intent.getAction() : null;
        if (TextUtils.isEmpty(action)) {
            flushTrackingInfo();
            flushVideoMaterials();
            return START_STICKY;
        }

        if (!ACTION_PUSH_CAPTURED_FILE.equals(action)) {
            sLogger.d("action: " + action);
        }

        if (ACTION_CREATE_VIDEO.equals(action)) {
            handleCreateVideo();

        } else if (ACTION_PUSH_CRASH.equals(action)) {
            CrashReport crash = (CrashReport) intent.getSerializableExtra(PARAM_CRASH);
            if (crash != null) {
                handlePushCrash(crash);
            }

        } else if (ACTION_APP_BECOME_BACKGROUND.equals(action)) {
            handleAppBackground();

        } else if (ACTION_APP_BECOME_FOREGROUND.equals(action)) {
            handleAppForeground();

        } else if (ACTION_PUSH_CAPTURED_FILE.equals(action)) {
            File capturedFile = (File) intent.getSerializableExtra(PARAM_CAPTURED_FILE);
            if (capturedFile != null && capturedFile.exists()) {
                handlePushCaptureFile(capturedFile);
            }

        } else if (ACTION_PUSH_HTTP_PACKET.equals(action)) {
            HttpPacket packet = (HttpPacket) intent.getSerializableExtra(PARAM_HTTP_PACKET);
            if (packet != null) {
                handlePushHttpPacket(packet);
            }
        }

        return START_STICKY;
    }

    private void handleCreateVideo() {
        String userId = mClient.getUser().getIdentifier();

        Calendar calendar = GregorianCalendar.getInstance();
        SimpleDateFormat format = new SimpleDateFormat(SolidConstants.TIME_FORMAT);
        String createdAt = format.format(calendar.getTime());

        VideoMaterial material = createVideoMaterial(userId, createdAt, null);
        if (material != null && !mVideoGenerator.hasVideoMaterial(material.getId())) {
            mVideoGenerator.pushVideoMaterial(material.getId(), material.getVideoFileName());
        }
    }

    private void handlePushCrash(CrashReport crash) {
        TrackingInfo.push(TrackingInfo.TYPE_CRASH, crash.toJson());
        flushTrackingInfo();

        VideoMaterial material =
                createVideoMaterial(SolidClient.getUser().getIdentifier(), crash.getCreatedAt(), crash.getUuid());

        if (material != null && !mVideoGenerator.hasVideoMaterial(material.getId())) {
            mVideoGenerator.pushVideoMaterial(material.getId(), material.getVideoFileName());
        }
    }

    private void handleAppBackground() {
        flushVideoMaterials();
        mVideoGenerator.setEnable();

        flushTrackingInfo();
    }

    private void handleAppForeground() {
        mVideoGenerator.setDisable();
    }

    private void handlePushCaptureFile(File file) {
        mFileManager.pushCapturedFile(file);
    }

    private void handlePushHttpPacket(HttpPacket packet) {
        TrackingInfo.push(TrackingInfo.TYPE_HTTP_PACKET, packet.toJson());
        flushTrackingInfo();
    }

    private VideoMaterial createVideoMaterial(String userId, String createdAt, String crashUUID) {
        VideoMaterial material = null;

        // Move all captured files to material dir
        try {
            material = VideoMaterial.create(this, userId, createdAt, crashUUID,
                    SolidConstants.VIDEO_MATERIAL_VIDEO_FILENAME_GIF);
            if (material == null) {
                return null;
            }

            File dir = VideoMaterial.getExternalStorage(this, material.getId());
            File videoFile = VideoMaterial.getVideoFile(this, material.getId(), material.getVideoFileName());

            // check video file already generated or not
            if (videoFile.exists()) {
                return material;
            }

            List<File> capturedFiles = new ArrayList<>();
            List<File> files = new ArrayList<>();
            List<File> popedFiles = mFileManager.popAll();

            if (popedFiles != null) {
                files.addAll(popedFiles);
            }

            for (File file : files) {
                if (!file.exists()) {
                    sLogger.d("file : " + file.getName() + " pushed but not exists");
                    continue;
                }
                File newFile = new File(dir, file.getName());
                FileUtils.move(file, newFile);
                capturedFiles.add(newFile);
            }
            return material;

        } catch (Throwable t) {
            t.printStackTrace();
            if (material != null) {
                VideoMaterial.deleteExternalStorage(this, material.getId());
                material.delete();
            }
        }
        return null;
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        if (mInfoSender != null) {
            mInfoSender.removeOnCrashSentListener(this);
        }

        if (mVideoGenerator != null) {
            mVideoGenerator.stop();
        }

        if (mConnectivityNotifier != null) {
            mConnectivityNotifier.removeSimpleObserver(this);
        }
    }

    @Override
    public void onVideoGenerated(long materialId, File file) {
        VideoMaterial material = VideoMaterial.findById(VideoMaterial.class, materialId);
        if (material != null && material.isReadyToUpload(mClient.getApplication())) {
            TrackingInfo.push(TrackingInfo.TYPE_VIDEO, material.toJson());
            material.delete();

            flushTrackingInfo();
        }
    }

    @Override
    public void onInternetConnected(boolean isWifi) {
        flushTrackingInfo();
    }

    @Override
    public void onInternetConnectTypeChanged(boolean isWifi) {
        flushTrackingInfo();
    }

    @Override
    public void onInternetDisconnected() {

    }

    @Override
    public void onCrashSent(JSONObject response) {
        try {
            List<VideoMaterial> updated = new ArrayList<>();

            if (response.optJSONObject("obj") != null) {
                JSONObject crashObj = response.getJSONObject("obj");
                String id = crashObj.getString("id");
                String uuid = crashObj.getJSONObject("data").getString("uuid");

                Map<String, VideoMaterial> materialMap = VideoMaterial.findByCrashUUID(uuid);
                VideoMaterial material = materialMap.get(uuid);
                if (material != null) {
                    material.setCrashId(id);
                    updated.add(material);
                } else {
                    sLogger.d("Failed to find crash video:" + uuid);
                }

            } else if (response.optJSONArray("objs") != null) {
                Map<String, String> idMap = new HashMap<>();

                JSONArray crashesObj = response.optJSONArray("objs");
                for (int i = 0; i < crashesObj.length(); i++) {
                    JSONObject crashObj = crashesObj.getJSONObject(i);
                    String id = crashObj.getString("id");
                    String uuid = crashObj.getJSONObject("data").getString("uuid");

                    idMap.put(uuid, id);
                }

                Set<String> uuids = idMap.keySet();

                Map<String, VideoMaterial> materialMap =
                        VideoMaterial.findByCrashUUID(uuids.toArray(new String[uuids.size()]));

                for (String uuid : idMap.keySet()) {
                    String crashId = idMap.get(uuid);
                    VideoMaterial material = materialMap.get(uuid);

                    if (material != null) {
                        material.setCrashId(crashId);
                        updated.add(material);
                    } else {
                        sLogger.d("Failed to find crash video:" + uuid);
                    }
                }
            }

            if (updated.size() > 0) {
                VideoMaterial.saveInTx(updated);

                for (VideoMaterial material : updated) {
                    if (material.isReadyToUpload(mClient.getApplication())) {
                        TrackingInfo.push(TrackingInfo.TYPE_VIDEO, material.toJson());
                        material.delete();
                    }
                }
                flushTrackingInfo();
            }

        } catch (JSONException e) {
            e.printStackTrace();
            sLogger.d("Failed to handle crash success response:" + response.toString());
        }
    }

    private void flushTrackingInfo() {
        mInfoSender.flush();
    }

    private void flushVideoMaterials() {
        List<VideoMaterial> materials = VideoMaterial.listLive(this);
        for (VideoMaterial material : materials) {

            boolean hasVideoFile = VideoMaterial.hasVideoFile(this, material.getId(), material.getVideoFileName());

            if (!hasVideoFile && !mVideoGenerator.hasVideoMaterial(material.getId())) {
                mVideoGenerator.pushVideoMaterial(material.getId(), material.getVideoFileName());

            } else if (material.isReadyToUpload(this)) {
                TrackingInfo.push(TrackingInfo.TYPE_VIDEO, material.toJson());
                material.delete();

            }
        }
    }
}
