package trip.spi;

import static trip.spi.helpers.filter.Filter.filter;
import static trip.spi.helpers.filter.Filter.first;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.ServiceConfigurationError;

import trip.spi.helpers.EmptyIterable;
import trip.spi.helpers.EmptyProviderContext;
import trip.spi.helpers.FieldQualifierExtractor;
import trip.spi.helpers.ProducerFactoryMap;
import trip.spi.helpers.ProvidableClass;
import trip.spi.helpers.QualifierExtractor;
import trip.spi.helpers.ServiceLoader;
import trip.spi.helpers.SingleObjectIterable;
import trip.spi.helpers.filter.AnyObject;
import trip.spi.helpers.filter.Condition;

@SuppressWarnings( { "rawtypes", "unchecked" } )
public class DefaultServiceProvider implements ServiceProvider {

	final Map<Class<?>, Iterable<Class<?>>> implementedClasses = new HashMap<>();
	final SingletonContext singletonContext = new SingletonContext();

	final Map<Class<?>, Iterable<?>> providers;
	final ProducerFactoryMap producers;

	public DefaultServiceProvider() {
		this.providers = createDefaultProvidedData();
		singletonContext.setQualifierExtractor( createQualifierExtractor() );
		runHookBeforeProducersAreReady();
		this.producers = loadAllProducers();
		runAllStartupListeners();
	}

	private QualifierExtractor createQualifierExtractor() {
		final Iterable<FieldQualifierExtractor> extractors = loadAll(FieldQualifierExtractor.class);
		return new QualifierExtractor( extractors );
	}

	private void runHookBeforeProducersAreReady() {
		final Iterable<StartupListener> startupListeners = loadAll( StartupListener.class );
		for ( final StartupListener listener : startupListeners )
			listener.beforeProducersReady( this );
	}

	private void runAllStartupListeners() {
		final Iterable<StartupListener> startupListeners = loadAll( StartupListener.class );
		for ( final StartupListener listener : startupListeners )
			listener.onStartup( this );
	}

	protected Map<Class<?>, Iterable<?>> createDefaultProvidedData() {
		final Map<Class<?>, Iterable<?>> injectables = new HashMap<Class<?>, Iterable<?>>();
		injectables.put( ServiceProvider.class, new SingleObjectIterable<DefaultServiceProvider>( this ) );
		return injectables;
	}

	protected ProducerFactoryMap loadAllProducers() {
		return ProducerFactoryMap.from( loadAll( ProducerFactory.class ) );
	}

	@Override
	public <T> T load( final Class<T> interfaceClazz ) {
		return load( interfaceClazz, AnyObject.instance() );
	}

	@Override
	public <T> T load( final Class<T> interfaceClazz, final Condition<T> condition ) {
		return load( interfaceClazz, condition, EmptyProviderContext.INSTANCE );
	}

	@Override
	public <T> T load( final Class<T> interfaceClazz, final ProviderContext context ) {
		return load( interfaceClazz, AnyObject.instance(), context );
	}

	@Override
	public <T> T load( final Class<T> interfaceClazz, final Condition<T> condition, final ProviderContext context )
			throws ServiceProviderException {
		final T produced = produceFromFactory( interfaceClazz, condition, context );
		if ( produced != null )
			return produced;
		return first( loadAll( interfaceClazz, condition ), condition );
	}

	@Override
	public <T> Iterable<T> loadAll( final Class<T> interfaceClazz, final Condition<T> condition ) {
		return filter( loadAll( interfaceClazz ), condition );
	}

	@Override
	public <T> Iterable<T> loadAll( final Class<T> interfaceClazz ) {
		Iterable<T> iterable = (Iterable<T>)this.providers.get( interfaceClazz );
		if ( iterable == null )
			synchronized ( providers ) {
				iterable = (Iterable<T>)this.providers.get( interfaceClazz );
				if ( iterable == null )
					iterable = loadAllServicesImplementingTheInterface( interfaceClazz );
			}
		return iterable;
	}

	private <T> Iterable<T> loadAllServicesImplementingTheInterface( final Class<T> interfaceClazz ) {
		try {
			return loadServiceFor( interfaceClazz );
		} catch ( final StackOverflowError cause ) {
			throw new ServiceConfigurationError(
				"Could not load implementations of " + interfaceClazz.getCanonicalName() +
					": Recursive dependency injection detected." );
		}
	}

	private <T> Iterable<T> loadServiceFor( final Class<T> interfaceClazz ) {
		final List<Class<T>> iterableInterfaces = loadClassesImplementing( interfaceClazz );
		Iterable<T> instances = null;
		if ( !iterableInterfaces.isEmpty() ){
			instances = singletonContext.instantiate( iterableInterfaces );
			provideOn( instances );
			providerFor( interfaceClazz, instances );
		} else {
			final T instance = singletonContext.instantiate( interfaceClazz );
			instances = instance == null ? EmptyIterable.instance() : new SingleObjectIterable<>( instance );
			provideOn( instances );
		}
		return instances;
	}

	public <T> List<Class<T>> loadClassesImplementing( final Class<T> interfaceClazz ) {
		List<Class<T>> implementations = (List)implementedClasses.get( interfaceClazz );
		if ( implementations == null )
			synchronized ( implementedClasses ) {
				implementations = (List)implementedClasses.get( interfaceClazz );
				if ( implementations == null ) {
					implementations = ServiceLoader.loadImplementationsFor( interfaceClazz );
					implementedClasses.put( (Class)interfaceClazz, (Iterable)implementations );
				}
			}
		return implementations;
	}

	@Override
	public <T> void providerFor( final Class<T> interfaceClazz, final ProducerFactory<T> provider ) {
			this.producers.memorizeProviderForClazz( provider, interfaceClazz );
	}

	@Override
	public <T> void providerFor( final Class<T> interfaceClazz, final T object ) {
		providerFor( interfaceClazz, new SingleObjectIterable<T>( object ) );
	}

	protected <T> void providerFor( final Class<T> interfaceClazz, final Iterable<T> iterable ) {
		this.providers.put( interfaceClazz, iterable );
	}

	@Override
	public <T> void provideOn( final Iterable<T> iterable ) {
		for ( final T object : iterable )
			provideOn( object );
	}

	@Override
	public void provideOn( final Object object ) {
		try {
			final ProvidableClass<?> providableClass = singletonContext.retrieveProvidableClass( object.getClass() );
			providableClass.provide( object, this );
		} catch ( final Exception cause ) {
			throw new ServiceProviderException( cause );
		}
	}

	private <T> T produceFromFactory( final Class<T> interfaceClazz, final Condition<T> condition, final ProviderContext context )
	{
		final ProducerFactory<T> provider = getProviderFor( interfaceClazz, condition );
		if ( provider != null )
			return provider.provide( context );
		return null;
	}

	public <T> ProducerFactory<T> getProviderFor( final Class<T> interfaceClazz, final Condition<T> condition ) {
		if ( this.producers == null )
			return null;
		return (ProducerFactory<T>)this.producers.get( interfaceClazz, condition );
	}

}
