/*
 * Decompiled with CFR 0.152.
 */
package io.serialized.client.feed;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import io.serialized.client.SerializedClientConfig;
import io.serialized.client.SerializedOkHttpClient;
import io.serialized.client.feed.Feed;
import io.serialized.client.feed.FeedEntry;
import io.serialized.client.feed.FeedEntryBatchHandler;
import io.serialized.client.feed.FeedEntryHandler;
import io.serialized.client.feed.FeedRequests;
import io.serialized.client.feed.FeedResponse;
import io.serialized.client.feed.FeedsResponse;
import io.serialized.client.feed.GetFeedRequest;
import io.serialized.client.feed.GetSequenceNumberRequest;
import io.serialized.client.feed.InMemorySequenceNumberTracker;
import io.serialized.client.feed.ListFeedsRequest;
import io.serialized.client.feed.RetryException;
import io.serialized.client.feed.SequenceNumberTracker;
import java.io.Closeable;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.logging.Logger;
import okhttp3.HttpUrl;
import okhttp3.OkHttpClient;
import okhttp3.Response;
import org.apache.commons.lang3.Validate;

public class FeedClient
implements Closeable {
    private static final String SEQUENCE_NUMBER_HEADER = "Serialized-SequenceNumber-Current";
    private final Logger logger = Logger.getLogger(this.getClass().getName());
    private final SerializedOkHttpClient client;
    private final HttpUrl apiRoot;
    private final Map<UUID, ExecutorService> executors = new ConcurrentHashMap<UUID, ExecutorService>();

    private FeedClient(Builder builder) {
        this.client = new SerializedOkHttpClient(builder.httpClient, builder.objectMapper);
        this.apiRoot = builder.apiRoot;
    }

    public static Builder feedClient(SerializedClientConfig config) {
        return new Builder(config);
    }

    @Override
    public void close() {
        this.executors.values().forEach(ExecutorService::shutdown);
    }

    public void close(UUID subscriptionId) {
        Optional.ofNullable(this.executors.remove(subscriptionId)).ifPresent(ExecutorService::shutdown);
    }

    public FeedResponse execute(GetFeedRequest request, long since) {
        HttpUrl.Builder urlBuilder = this.url(request.feedName);
        Optional.ofNullable(request.limit).ifPresent(limit -> urlBuilder.addQueryParameter("limit", String.valueOf(limit)));
        Optional.ofNullable(request.partitionCount).ifPresent(pCount -> urlBuilder.addQueryParameter("partitionCount", String.valueOf(pCount)));
        Optional.ofNullable(request.partitionNumber).ifPresent(pNumber -> urlBuilder.addQueryParameter("partitionNumber", String.valueOf(pNumber)));
        Optional.ofNullable(request.waitTime).ifPresent(pNumber -> urlBuilder.addQueryParameter("waitTime", String.valueOf(pNumber.toMillis())));
        for (String type : request.types) {
            urlBuilder.addQueryParameter("filterType", type);
        }
        HttpUrl url = urlBuilder.addQueryParameter("since", String.valueOf(since)).build();
        if (request.tenantId().isPresent()) {
            return this.client.get(url, FeedResponse.class, request.tenantId);
        }
        return this.client.get(url, FeedResponse.class);
    }

    public UUID subscribe(GetFeedRequest request, FeedEntryHandler feedEntryHandler) {
        return this.subscribe(request, (SequenceNumberTracker)new InMemorySequenceNumberTracker(), feedEntryHandler);
    }

    public UUID subscribe(GetFeedRequest request, FeedEntryBatchHandler feedEntryBatchHandler) {
        return this.subscribe(request, (SequenceNumberTracker)new InMemorySequenceNumberTracker(), feedEntryBatchHandler);
    }

    public UUID subscribe(GetFeedRequest request, SequenceNumberTracker sequenceNumberTracker, FeedEntryHandler feedEntryHandler) {
        Validate.isTrue((request.waitTime.getSeconds() > 0L ? 1 : 0) != 0, (String)"'waitTime' in request cannot be zero when subscribing to a feed", (Object[])new Object[0]);
        ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor();
        if (request.startFromHead) {
            long sequenceNumber = this.execute(FeedRequests.getSequenceNumber().withFeed(request.feedName).build());
            sequenceNumberTracker.updateLastConsumedSequenceNumber(sequenceNumber);
        }
        executor.scheduleWithFixedDelay(() -> {
            try {
                FeedResponse response;
                do {
                    long sequenceNumber = sequenceNumberTracker.lastConsumedSequenceNumber();
                    response = this.execute(request, sequenceNumber);
                    if (sequenceNumber > sequenceNumberTracker.lastConsumedSequenceNumber()) {
                        return;
                    }
                    if (response.entries().isEmpty() && response.currentSequenceNumber() > sequenceNumber) {
                        sequenceNumberTracker.updateLastConsumedSequenceNumber(response.currentSequenceNumber());
                        continue;
                    }
                    for (FeedEntry feedEntry : response.entries()) {
                        try {
                            feedEntryHandler.handle(feedEntry);
                            try {
                                sequenceNumberTracker.updateLastConsumedSequenceNumber(feedEntry.sequenceNumber());
                            }
                            catch (RuntimeException re) {
                                this.logger.log(Level.WARNING, String.format("Error updating sequence number after processing: %s - last polled number was [%d]", feedEntry, sequenceNumber), re);
                                throw re;
                            }
                        }
                        catch (RetryException retryException) {
                        }
                    }
                } while (request.eagerFetching && response.hasMore());
            }
            catch (Exception exception) {
                this.logger.log(Level.WARNING, String.format("Error polling event feed [%s]: %s", request.feedName, exception.getMessage()), exception);
                try {
                    Thread.sleep(1000L);
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
            }
        }, 1L, 1L, TimeUnit.MILLISECONDS);
        UUID subscriptionId = UUID.randomUUID();
        this.executors.put(subscriptionId, executor);
        return subscriptionId;
    }

    public UUID subscribe(GetFeedRequest request, SequenceNumberTracker sequenceNumberTracker, FeedEntryBatchHandler feedEntryBatchHandler) {
        Validate.isTrue((request.waitTime.getSeconds() > 0L ? 1 : 0) != 0, (String)"'waitTime' in request cannot be zero when subscribing to a feed", (Object[])new Object[0]);
        ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor();
        if (request.startFromHead) {
            long sequenceNumber = this.execute(FeedRequests.getSequenceNumber().withFeed(request.feedName).build());
            sequenceNumberTracker.updateLastConsumedSequenceNumber(sequenceNumber);
        }
        executor.scheduleWithFixedDelay(() -> {
            try {
                FeedResponse response;
                do {
                    long sequenceNumber = sequenceNumberTracker.lastConsumedSequenceNumber();
                    response = this.execute(request, sequenceNumber);
                    if (sequenceNumber > sequenceNumberTracker.lastConsumedSequenceNumber()) {
                        return;
                    }
                    List<FeedEntry> entries = response.entries();
                    if (entries.isEmpty()) {
                        if (response.currentSequenceNumber() <= sequenceNumber) continue;
                        sequenceNumberTracker.updateLastConsumedSequenceNumber(response.currentSequenceNumber());
                        continue;
                    }
                    feedEntryBatchHandler.handle(entries);
                    sequenceNumberTracker.updateLastConsumedSequenceNumber(entries.get(entries.size() - 1).sequenceNumber());
                } while (request.eagerFetching && response.hasMore());
            }
            catch (Exception exception) {
                this.logger.log(Level.WARNING, String.format("Error polling event feed [%s]: %s", request.feedName, exception.getMessage()), exception);
                try {
                    Thread.sleep(1000L);
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
            }
        }, 1L, 1L, TimeUnit.MILLISECONDS);
        UUID subscriptionId = UUID.randomUUID();
        this.executors.put(subscriptionId, executor);
        return subscriptionId;
    }

    public List<Feed> execute(ListFeedsRequest request) {
        HttpUrl url = this.apiRoot.newBuilder().addPathSegment("feeds").build();
        if (request.tenantId().isPresent()) {
            return this.client.get(url, FeedsResponse.class, request.tenantId).feeds();
        }
        return this.client.get(url, FeedsResponse.class).feeds();
    }

    public long execute(GetSequenceNumberRequest request) {
        HttpUrl url = this.url(request.feedName).build();
        Function<Response, Long> func = response -> Long.parseLong(Objects.requireNonNull(response.header(SEQUENCE_NUMBER_HEADER)));
        if (request.tenantId().isPresent()) {
            return this.client.head(url, func, request.tenantId);
        }
        return this.client.head(url, func);
    }

    private HttpUrl.Builder url(String feedName) {
        Validate.notBlank((CharSequence)feedName, (String)"No feed specified", (Object[])new Object[0]);
        return this.apiRoot.newBuilder().addPathSegment("feeds").addPathSegment(feedName);
    }

    public static class Builder {
        private final ObjectMapper objectMapper = new ObjectMapper().disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES).disable(SerializationFeature.FAIL_ON_EMPTY_BEANS).setVisibility(PropertyAccessor.FIELD, JsonAutoDetect.Visibility.ANY).setSerializationInclusion(JsonInclude.Include.NON_NULL);
        private final OkHttpClient httpClient;
        private final HttpUrl apiRoot;

        public Builder(SerializedClientConfig config) {
            this.httpClient = config.newHttpClient();
            this.apiRoot = config.apiRoot();
        }

        public Builder configureObjectMapper(Consumer<ObjectMapper> consumer) {
            consumer.accept(this.objectMapper);
            return this;
        }

        public FeedClient build() {
            return new FeedClient(this);
        }
    }
}

