/*
 * Decompiled with CFR 0.152.
 */
package io.serialized.client.aggregate;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import io.serialized.client.ApiException;
import io.serialized.client.ConcurrencyException;
import io.serialized.client.InvalidRequestException;
import io.serialized.client.SerializedClientConfig;
import io.serialized.client.SerializedOkHttpClient;
import io.serialized.client.aggregate.AggregateBulkRequest;
import io.serialized.client.aggregate.AggregateDelete;
import io.serialized.client.aggregate.AggregateDeleteConfirmation;
import io.serialized.client.aggregate.AggregateExists;
import io.serialized.client.aggregate.AggregateRequest;
import io.serialized.client.aggregate.AggregateUpdate;
import io.serialized.client.aggregate.BulkSaveEvents;
import io.serialized.client.aggregate.Event;
import io.serialized.client.aggregate.EventBatch;
import io.serialized.client.aggregate.EventDeserializer;
import io.serialized.client.aggregate.EventHandler;
import io.serialized.client.aggregate.RetryStrategy;
import io.serialized.client.aggregate.StateBuilder;
import io.serialized.client.aggregate.UpdateStrategy;
import io.serialized.client.aggregate.cache.StateCache;
import io.serialized.client.aggregate.cache.VersionedState;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.logging.Logger;
import okhttp3.HttpUrl;
import okhttp3.OkHttpClient;
import okhttp3.Response;
import org.apache.commons.lang3.Validate;

public class AggregateClient<T> {
    private final Logger logger = Logger.getLogger(this.getClass().getName());
    private final SerializedOkHttpClient client;
    private final HttpUrl apiRoot;
    private final StateBuilder<T> stateBuilder;
    private final String aggregateType;
    private final RetryStrategy retryStrategy;
    private final int limit;

    private AggregateClient(Builder<T> builder) {
        this.client = new SerializedOkHttpClient(((Builder)builder).httpClient, ((Builder)builder).objectMapper);
        this.apiRoot = ((Builder)builder).apiRoot;
        this.aggregateType = ((Builder)builder).aggregateType;
        this.stateBuilder = ((Builder)builder).stateBuilder;
        this.retryStrategy = ((Builder)builder).retryStrategy;
        this.limit = ((Builder)builder).limit;
    }

    public static <T> Builder<T> aggregateClient(String aggregateType, Class<T> stateClass, SerializedClientConfig config) {
        return new Builder<T>(aggregateType, stateClass, config);
    }

    public void save(AggregateRequest request) {
        try {
            HttpUrl url = this.getAggregateUrl(request.aggregateId).addPathSegment("events").build();
            if (request.tenantId().isPresent()) {
                UUID tenantId = request.tenantId().get();
                this.client.post(url, request.eventBatch(), tenantId);
            } else {
                this.client.post(url, request.eventBatch());
            }
        }
        catch (ApiException e) {
            this.handleConcurrencyException(e);
        }
    }

    public void save(AggregateBulkRequest request) {
        try {
            HttpUrl url = this.getAggregateTypeUrl().addPathSegment("events").build();
            BulkSaveEvents payload = request.eventBatches();
            if (request.tenantId().isPresent()) {
                UUID tenantId = request.tenantId().get();
                this.client.post(url, payload, tenantId);
            } else {
                this.client.post(url, payload);
            }
        }
        catch (ApiException e) {
            this.handleConcurrencyException(e);
        }
    }

    public int update(String aggregateId, AggregateUpdate<T> update) {
        return this.update(UUID.fromString(aggregateId), update);
    }

    public int update(UUID aggregateId, AggregateUpdate<T> update) {
        ConcurrencyException lastException = new ConcurrencyException(409, "Conflict");
        for (int i = 0; i <= this.retryStrategy.getRetryCount(); ++i) {
            try {
                return this.updateInternal(aggregateId, update, eventBatch -> this.storeBatch(aggregateId, update.tenantId(), (EventBatch)eventBatch));
            }
            catch (ConcurrencyException concurrencyException) {
                lastException = concurrencyException;
                try {
                    Thread.sleep(this.retryStrategy.getSleepMs());
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
                continue;
            }
        }
        throw lastException;
    }

    public int bulkUpdate(Set<UUID> aggregateIds, AggregateUpdate<T> update) {
        ConcurrencyException lastException = new ConcurrencyException(409, "Conflict");
        for (int i = 0; i <= this.retryStrategy.getRetryCount(); ++i) {
            try {
                ArrayList<EventBatch> batches = new ArrayList<EventBatch>();
                for (UUID aggregateId : aggregateIds) {
                    this.updateInternal(aggregateId, update, eventBatch -> {
                        if (!eventBatch.events().isEmpty()) {
                            batches.add(eventBatch.withAggregateId(aggregateId));
                        }
                        return eventBatch.events().size();
                    });
                }
                return this.storeBulk(update.tenantId(), batches);
            }
            catch (ConcurrencyException concurrencyException) {
                lastException = concurrencyException;
                try {
                    Thread.sleep(this.retryStrategy.getSleepMs());
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
                continue;
            }
        }
        throw lastException;
    }

    private int updateInternal(UUID aggregateId, AggregateUpdate<T> update, Function<EventBatch, Integer> onSave) {
        this.assertValidUpdateConfig(update);
        if (update.stateCache().isPresent()) {
            T currentState;
            int currentVersion;
            StateCache<T> stateCache = update.stateCache().get();
            Optional<VersionedState<T>> cachedState = stateCache.get(aggregateId);
            if (cachedState.isPresent()) {
                VersionedState<T> versionedState = cachedState.get();
                currentVersion = versionedState.version();
                currentState = versionedState.state();
            } else {
                LoadAggregateResponse aggregateResponse = this.loadState(aggregateId, update.tenantId());
                currentVersion = aggregateResponse.aggregateVersion;
                currentState = this.stateBuilder.buildState(aggregateResponse.events);
            }
            try {
                List<Event<?>> events = update.apply(currentState);
                if (events.size() >= 64) {
                    throw new InvalidRequestException(String.format("Cannot store more than %d events per batch", 64));
                }
                int eventStored = onSave.apply(new EventBatch(events, currentVersion));
                if (eventStored > 0) {
                    stateCache.put(aggregateId, new VersionedState<T>(this.stateBuilder.buildState(currentState, events), currentVersion + 1));
                }
                return eventStored;
            }
            catch (ConcurrencyException e) {
                this.logger.log(Level.INFO, String.format("Concurrency exception detected - invalidating cached entry with ID [%s]", aggregateId.toString()));
                stateCache.invalidate(aggregateId);
                throw e;
            }
        }
        LoadAggregateResponse aggregateResponse = this.loadState(aggregateId, update.tenantId());
        T state = this.stateBuilder.buildState(aggregateResponse.events);
        Integer expectedVersion = update.useOptimisticConcurrencyOnUpdate() ? Integer.valueOf(aggregateResponse.aggregateVersion) : null;
        List<Event<?>> events = update.apply(state);
        if (events.size() >= 64) {
            throw new InvalidRequestException(String.format("Cannot store more than %d events per batch", 64));
        }
        return onSave.apply(new EventBatch(events, expectedVersion));
    }

    private void assertValidUpdateConfig(AggregateUpdate<T> update) {
        if (update.stateCache().isPresent() && !update.useOptimisticConcurrencyOnUpdate()) {
            throw new IllegalArgumentException("Cannot use stateCache with optimisticConcurrencyOnUpdate disabled");
        }
    }

    public AggregateDeleteConfirmation delete(AggregateDelete delete) {
        if (delete.aggregateId == null) {
            return this.getDeleteToken(this.getAggregateTypeUrl(), delete.tenantId);
        }
        return this.getDeleteToken(this.getAggregateUrl(delete.aggregateId), delete.tenantId);
    }

    public boolean exists(AggregateExists exists) {
        try {
            HttpUrl url = this.getAggregateUrl(exists.aggregateId).build();
            if (exists.tenantId == null) {
                return this.client.head(url, Response::code) == 200;
            }
            return this.client.head(url, Response::code, exists.tenantId) == 200;
        }
        catch (ApiException e) {
            if (e.statusCode() == 404) {
                return false;
            }
            throw e;
        }
    }

    private AggregateDeleteConfirmation getDeleteToken(HttpUrl.Builder urlBuilder, UUID tenantId) {
        if (tenantId == null) {
            HttpUrl deleteAggregateUrl = this.extractDeleteToken(urlBuilder, this.client.delete(urlBuilder.build(), Map.class));
            return new AggregateDeleteConfirmation(this.client, deleteAggregateUrl);
        }
        HttpUrl deleteAggregateUrl = this.extractDeleteToken(urlBuilder, this.client.delete(urlBuilder.build(), Map.class, tenantId));
        return new AggregateDeleteConfirmation(this.client, deleteAggregateUrl, tenantId);
    }

    private HttpUrl extractDeleteToken(HttpUrl.Builder urlBuilder, Map<String, String> deleteResponse) {
        return urlBuilder.addQueryParameter("deleteToken", deleteResponse.get("deleteToken")).build();
    }

    private LoadAggregateResponse loadState(UUID aggregateId, Optional<UUID> tenantId) {
        HttpUrl.Builder builder = this.getAggregateUrl(aggregateId).addQueryParameter("limit", String.valueOf(this.limit));
        int since = 0;
        LoadAggregateResponse response = new LoadAggregateResponse();
        if (tenantId.isPresent()) {
            do {
                HttpUrl url = builder.setQueryParameter("since", String.valueOf(since)).build();
                response.merge(this.client.get(url, LoadAggregateResponse.class, tenantId.get()));
                since += this.limit;
            } while (response.hasMore);
        } else {
            do {
                HttpUrl url = builder.setQueryParameter("since", String.valueOf(since)).build();
                response.merge(this.client.get(url, LoadAggregateResponse.class));
                since += this.limit;
            } while (response.hasMore);
        }
        return response;
    }

    private int storeBatch(UUID aggregateId, Optional<UUID> tenantId, EventBatch eventBatch) {
        int eventCount = eventBatch.events().size();
        if (eventCount == 0) {
            return 0;
        }
        try {
            HttpUrl url = this.getAggregateUrl(aggregateId).addPathSegment("events").build();
            if (tenantId.isPresent()) {
                this.client.post(url, eventBatch, tenantId.get());
            } else {
                this.client.post(url, eventBatch);
            }
        }
        catch (ApiException e) {
            this.handleConcurrencyException(e);
        }
        return eventCount;
    }

    private int storeBulk(Optional<UUID> tenantId, List<EventBatch> batches) {
        if (batches.isEmpty()) {
            return 0;
        }
        try {
            HttpUrl url = this.getAggregateTypeUrl().addPathSegment("events").build();
            if (tenantId.isPresent()) {
                this.client.post(url, BulkSaveEvents.newBulkSaveEvents(batches), tenantId.get());
            } else {
                this.client.post(url, BulkSaveEvents.newBulkSaveEvents(batches));
            }
        }
        catch (ApiException e) {
            this.handleConcurrencyException(e);
        }
        return batches.stream().map(EventBatch::events).mapToInt(List::size).sum();
    }

    private void handleConcurrencyException(ApiException e) {
        if (e.statusCode() == 409) {
            throw new ConcurrencyException(409, e.getMessage());
        }
        throw e;
    }

    private HttpUrl.Builder getAggregateTypeUrl() {
        return this.apiRoot.newBuilder().addPathSegment("aggregates").addPathSegment(this.aggregateType);
    }

    private HttpUrl.Builder getAggregateUrl(UUID aggregateId) {
        return this.getAggregateTypeUrl().addPathSegment(aggregateId.toString());
    }

    private static class LoadAggregateResponse {
        String aggregateId;
        String aggregateType;
        int aggregateVersion;
        List<Event<?>> events;
        boolean hasMore;

        private LoadAggregateResponse() {
        }

        public void merge(LoadAggregateResponse response) {
            this.aggregateId = response.aggregateId;
            this.aggregateType = response.aggregateType;
            this.aggregateVersion = response.aggregateVersion;
            if (this.events == null) {
                this.events = new ArrayList(response.events);
            } else {
                this.events.addAll(response.events);
            }
            this.hasMore = response.hasMore;
        }
    }

    public static class Builder<T> {
        private final ObjectMapper objectMapper = new ObjectMapper().disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES).disable(SerializationFeature.FAIL_ON_EMPTY_BEANS).setVisibility(PropertyAccessor.FIELD, JsonAutoDetect.Visibility.ANY).setSerializationInclusion(JsonInclude.Include.NON_NULL);
        private final HttpUrl apiRoot;
        private final OkHttpClient httpClient;
        private final StateBuilder<T> stateBuilder;
        private final String aggregateType;
        private final Map<String, Class> eventTypes = new HashMap<String, Class>();
        private RetryStrategy retryStrategy = RetryStrategy.DEFAULT;
        private UpdateStrategy updateStrategy = UpdateStrategy.DEFAULT;
        private int limit = 1000;

        Builder(String aggregateType, Class<T> stateClass, SerializedClientConfig config) {
            this.aggregateType = aggregateType;
            this.apiRoot = config.apiRoot();
            this.httpClient = config.newHttpClient();
            this.stateBuilder = StateBuilder.stateBuilder(stateClass);
        }

        public <E> Builder<T> registerHandler(Class<E> eventClass, EventHandler<T, E> handler) {
            return this.registerHandler(eventClass.getSimpleName(), eventClass, handler);
        }

        public <E> Builder<T> registerHandler(String eventType, Class<E> eventClass, EventHandler<T, E> handler) {
            this.eventTypes.put(eventType, eventClass);
            this.stateBuilder.withHandler(eventClass, handler);
            return this;
        }

        public <E> Builder<T> withRetryStrategy(RetryStrategy retryStrategy) {
            this.retryStrategy = retryStrategy;
            return this;
        }

        public <E> Builder<T> withUpdateStrategy(UpdateStrategy updateStrategy) {
            this.updateStrategy = updateStrategy;
            return this;
        }

        public <E> Builder<T> withLimit(int limit) {
            this.limit = limit;
            return this;
        }

        public <E> Builder<T> configureObjectMapper(Consumer<ObjectMapper> consumer) {
            consumer.accept(this.objectMapper);
            return this;
        }

        public AggregateClient<T> build() {
            Validate.notNull((Object)this.aggregateType, (String)"'aggregateType' must be set", (Object[])new Object[0]);
            this.objectMapper.registerModule(EventDeserializer.module(this.eventTypes));
            this.stateBuilder.setFailOnMissingHandler(this.updateStrategy.failOnMissingHandler());
            this.stateBuilder.setIgnoredEventTypes(this.updateStrategy.ignoredEventTypes());
            return new AggregateClient(this);
        }
    }
}

