/*
 * Decompiled with CFR 0.152.
 */
package io.serialized.client.aggregate;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import io.serialized.client.ApiException;
import io.serialized.client.ConcurrencyException;
import io.serialized.client.SerializedClientConfig;
import io.serialized.client.SerializedOkHttpClient;
import io.serialized.client.aggregate.AggregateDelete;
import io.serialized.client.aggregate.AggregateDeleteConfirmation;
import io.serialized.client.aggregate.AggregateExists;
import io.serialized.client.aggregate.AggregateRequest;
import io.serialized.client.aggregate.AggregateUpdate;
import io.serialized.client.aggregate.Event;
import io.serialized.client.aggregate.EventBatch;
import io.serialized.client.aggregate.EventDeserializer;
import io.serialized.client.aggregate.EventHandler;
import io.serialized.client.aggregate.RetryStrategy;
import io.serialized.client.aggregate.StateBuilder;
import io.serialized.client.aggregate.UpdateStrategy;
import io.serialized.client.aggregate.cache.StateCache;
import io.serialized.client.aggregate.cache.VersionedState;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.function.Consumer;
import okhttp3.HttpUrl;
import okhttp3.OkHttpClient;
import okhttp3.Response;
import org.apache.commons.lang3.Validate;

public class AggregateClient<T> {
    private final SerializedOkHttpClient client;
    private final HttpUrl apiRoot;
    private final StateBuilder<T> stateBuilder;
    private final String aggregateType;
    private final RetryStrategy retryStrategy;

    private AggregateClient(Builder<T> builder) {
        this.client = new SerializedOkHttpClient(((Builder)builder).httpClient, ((Builder)builder).objectMapper);
        this.apiRoot = ((Builder)builder).apiRoot;
        this.aggregateType = ((Builder)builder).aggregateType;
        this.stateBuilder = ((Builder)builder).stateBuilder;
        this.retryStrategy = ((Builder)builder).retryStrategy;
    }

    public static <T> Builder<T> aggregateClient(String aggregateType, Class<T> stateClass, SerializedClientConfig config) {
        return new Builder<T>(aggregateType, stateClass, config);
    }

    public void save(AggregateRequest request) {
        try {
            HttpUrl url = this.getAggregateUrl(request.aggregateId).addPathSegment("events").build();
            if (request.tenantId().isPresent()) {
                UUID tenantId = request.tenantId().get();
                this.client.post(url, request.eventBatch(), tenantId);
            } else {
                this.client.post(url, request.eventBatch());
            }
        }
        catch (ApiException e) {
            this.handleConcurrencyException(e);
        }
    }

    public int update(String aggregateId, AggregateUpdate<T> update) {
        return this.update(UUID.fromString(aggregateId), update);
    }

    public int update(UUID aggregateId, AggregateUpdate<T> update) {
        ConcurrencyException lastException = new ConcurrencyException(409, "Conflict");
        for (int i = 0; i <= this.retryStrategy.getRetryCount(); ++i) {
            try {
                return this.updateInternal(aggregateId, update);
            }
            catch (ConcurrencyException concurrencyException) {
                lastException = concurrencyException;
                try {
                    Thread.sleep(this.retryStrategy.getSleepMs());
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
                continue;
            }
        }
        throw lastException;
    }

    private int updateInternal(UUID aggregateId, AggregateUpdate<T> update) {
        this.assertValidUpdateConfig(update);
        if (update.stateCache().isPresent()) {
            T currentState;
            long currentVersion;
            StateCache<T> stateCache = update.stateCache().get();
            Optional<VersionedState<T>> cachedState = stateCache.get(aggregateId);
            if (cachedState.isPresent()) {
                VersionedState<T> versionedState = cachedState.get();
                currentVersion = versionedState.version();
                currentState = versionedState.state();
            } else {
                LoadAggregateResponse aggregateResponse = this.loadState(aggregateId, update.tenantId());
                currentVersion = aggregateResponse.aggregateVersion;
                currentState = this.stateBuilder.buildState(aggregateResponse.events);
            }
            try {
                List<Event<?>> events = update.apply(currentState);
                int eventStored = this.storeBatch(aggregateId, update.tenantId(), new EventBatch(events, currentVersion));
                if (eventStored > 0) {
                    stateCache.put(aggregateId, new VersionedState<T>(this.stateBuilder.buildState(currentState, events), currentVersion + 1L));
                }
                return eventStored;
            }
            catch (ConcurrencyException e) {
                stateCache.invalidate(aggregateId);
                throw e;
            }
        }
        LoadAggregateResponse aggregateResponse = this.loadState(aggregateId, update.tenantId());
        T state = this.stateBuilder.buildState(aggregateResponse.events);
        Long expectedVersion = update.useOptimisticConcurrencyOnUpdate() ? Long.valueOf(aggregateResponse.aggregateVersion) : null;
        List<Event<?>> events = update.apply(state);
        return this.storeBatch(aggregateId, update.tenantId(), new EventBatch(events, expectedVersion));
    }

    private void assertValidUpdateConfig(AggregateUpdate<T> update) {
        if (update.stateCache().isPresent() && !update.useOptimisticConcurrencyOnUpdate()) {
            throw new IllegalArgumentException("Cannot use stateCache with optimisticConcurrencyOnUpdate disabled");
        }
    }

    public AggregateDeleteConfirmation delete(AggregateDelete delete) {
        if (delete.aggregateId == null) {
            return this.getDeleteToken(this.getAggregateTypeUrl(), delete.tenantId);
        }
        return this.getDeleteToken(this.getAggregateUrl(delete.aggregateId), delete.tenantId);
    }

    public boolean exists(AggregateExists exists) {
        try {
            HttpUrl url = this.getAggregateUrl(exists.aggregateId).build();
            if (exists.tenantId == null) {
                return this.client.head(url, Response::code) == 200;
            }
            return this.client.head(url, Response::code, exists.tenantId) == 200;
        }
        catch (ApiException e) {
            if (e.statusCode() == 404) {
                return false;
            }
            throw e;
        }
    }

    private AggregateDeleteConfirmation getDeleteToken(HttpUrl.Builder urlBuilder, UUID tenantId) {
        if (tenantId == null) {
            return this.extractDeleteToken(urlBuilder, this.client.delete(urlBuilder.build(), Map.class));
        }
        return this.extractDeleteToken(urlBuilder, this.client.delete(urlBuilder.build(), Map.class, tenantId));
    }

    private AggregateDeleteConfirmation extractDeleteToken(HttpUrl.Builder urlBuilder, Map<String, String> deleteResponse) {
        String deleteToken = deleteResponse.get("deleteToken");
        HttpUrl deleteAggregateUrl = urlBuilder.addQueryParameter("deleteToken", deleteToken).build();
        return new AggregateDeleteConfirmation(this.client, deleteAggregateUrl);
    }

    private LoadAggregateResponse loadState(UUID aggregateId, Optional<UUID> tenantId) {
        HttpUrl url = this.getAggregateUrl(aggregateId).build();
        if (tenantId.isPresent()) {
            return this.client.get(url, LoadAggregateResponse.class, tenantId.get());
        }
        return this.client.get(url, LoadAggregateResponse.class);
    }

    private int storeBatch(UUID aggregateId, Optional<UUID> tenantId, EventBatch eventBatch) {
        if (eventBatch.events().isEmpty()) {
            return 0;
        }
        try {
            HttpUrl url = this.getAggregateUrl(aggregateId).addPathSegment("events").build();
            if (tenantId.isPresent()) {
                this.client.post(url, eventBatch, tenantId.get());
            } else {
                this.client.post(url, eventBatch);
            }
        }
        catch (ApiException e) {
            this.handleConcurrencyException(e);
        }
        return eventBatch.events().size();
    }

    private void handleConcurrencyException(ApiException e) {
        if (e.statusCode() == 409) {
            throw new ConcurrencyException(409, e.getMessage());
        }
        throw e;
    }

    private HttpUrl.Builder getAggregateTypeUrl() {
        return this.apiRoot.newBuilder().addPathSegment("aggregates").addPathSegment(this.aggregateType);
    }

    private HttpUrl.Builder getAggregateUrl(UUID aggregateId) {
        return this.getAggregateTypeUrl().addPathSegment(aggregateId.toString());
    }

    private static class LoadAggregateResponse {
        String aggregateId;
        String aggregateType;
        long aggregateVersion;
        List<Event<?>> events;

        private LoadAggregateResponse() {
        }
    }

    public static class Builder<T> {
        private final ObjectMapper objectMapper = new ObjectMapper().disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES).disable(SerializationFeature.FAIL_ON_EMPTY_BEANS).setVisibility(PropertyAccessor.FIELD, JsonAutoDetect.Visibility.ANY).setSerializationInclusion(JsonInclude.Include.NON_NULL);
        private final HttpUrl apiRoot;
        private final OkHttpClient httpClient;
        private final StateBuilder<T> stateBuilder;
        private final String aggregateType;
        private final Map<String, Class> eventTypes = new HashMap<String, Class>();
        private RetryStrategy retryStrategy = RetryStrategy.DEFAULT;
        private UpdateStrategy updateStrategy = UpdateStrategy.DEFAULT;

        Builder(String aggregateType, Class<T> stateClass, SerializedClientConfig config) {
            this.aggregateType = aggregateType;
            this.apiRoot = config.apiRoot();
            this.httpClient = config.httpClient();
            this.stateBuilder = StateBuilder.stateBuilder(stateClass);
        }

        public <E> Builder<T> registerHandler(Class<E> eventClass, EventHandler<T, E> handler) {
            return this.registerHandler(eventClass.getSimpleName(), eventClass, handler);
        }

        public <E> Builder<T> registerHandler(String eventType, Class<E> eventClass, EventHandler<T, E> handler) {
            this.eventTypes.put(eventType, eventClass);
            this.stateBuilder.withHandler(eventClass, handler);
            return this;
        }

        public <E> Builder<T> withRetryStrategy(RetryStrategy retryStrategy) {
            this.retryStrategy = retryStrategy;
            return this;
        }

        public <E> Builder<T> withUpdateStrategy(UpdateStrategy updateStrategy) {
            this.updateStrategy = updateStrategy;
            return this;
        }

        public <E> Builder<T> configureObjectMapper(Consumer<ObjectMapper> consumer) {
            consumer.accept(this.objectMapper);
            return this;
        }

        public AggregateClient<T> build() {
            Validate.notNull((Object)this.aggregateType, (String)"'aggregateType' must be set", (Object[])new Object[0]);
            this.objectMapper.registerModule(EventDeserializer.module(this.eventTypes));
            this.stateBuilder.setFailOnMissingHandler(this.updateStrategy.failOnMissingHandler());
            this.stateBuilder.setIgnoredEventTypes(this.updateStrategy.ignoredEventTypes());
            return new AggregateClient(this);
        }
    }
}

