package io.rudin.webdoc.engine.handlebars;

import io.rudin.webdoc.api.TemplateEngine;
import io.rudin.webdoc.api.WebdocContext;
import io.rudin.webdoc.engine.handlebars.helpers.CodeHelper;
import io.rudin.webdoc.engine.handlebars.helpers.IncludeHelper;
import io.rudin.webdoc.util.StreamUtil;

import java.io.IOException;
import java.io.InputStream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.github.jknack.handlebars.Handlebars;
import com.github.jknack.handlebars.Template;
import com.github.jknack.handlebars.io.AbstractTemplateLoader;
import com.github.jknack.handlebars.io.TemplateSource;

public class HandlebarsEngine implements TemplateEngine
{
	private final Handlebars handlebars = new Handlebars(new ClasspathPartialLoader());

	
	public HandlebarsEngine()
	{
		handlebars.registerHelper("include", new IncludeHelper());
		handlebars.registerHelper("code", new CodeHelper(this));
	}
	
	@Override
	public String apply(String template, WebdocContext ctx) throws Exception
	{
		this.ctx = ctx;
		Template tmpl = handlebars.compileInline(template);
		return tmpl.apply(ctx.getContext());
	}
	
	private WebdocContext ctx;
	
	public WebdocContext getContext()
	{
		return ctx;
	}
	
	
	static class ClasspathPartialLoader extends AbstractTemplateLoader
	{

		private static final Logger logger = LoggerFactory.getLogger(ClasspathPartialLoader.class);
		
		@Override
		public TemplateSource sourceAt(String location) throws IOException {
			
			logger.debug("partial sourceAt: {}", location);
			
			return new ClasspathTemplateSource(location);
		}
		
	}
	
	static class ClasspathTemplateSource implements TemplateSource
	{

		public ClasspathTemplateSource(String filename) throws IOException
		{
			this.filename = filename;
			this.mtime = System.currentTimeMillis();
			
			InputStream inputStream = HandlebarsEngine.class.getResourceAsStream("/" + filename);
			this.content = StreamUtil.streamToString(inputStream);
		}
		
		private final String content, filename;
		
		private final long mtime;
		
		@Override
		public String content() throws IOException {
			return content;
		}

		@Override
		public String filename() {
			return filename;
		}

		@Override
		public long lastModified() {
			return mtime;
		}
		
	}
	

	

}
