package io.rudin.webdoc.core.template;

import io.rudin.webdoc.api.RawTransformer;
import io.rudin.webdoc.api.TemplateEngine;
import io.rudin.webdoc.api.Transformer;
import io.rudin.webdoc.api.WebdocContext;
import io.rudin.webdoc.core.callback.TransformerCallback;
import io.rudin.webdoc.core.spi.RawTransformerLoader;
import io.rudin.webdoc.core.spi.TemplateEngineLoader;
import io.rudin.webdoc.core.spi.TransformerLoader;
import io.rudin.webdoc.util.StreamUtil;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * http://stackoverflow.com/questions/6214703/copy-entire-directory-contents-to-another-directory
 * @author user
 *
 */
public class TransformFileVisitor extends SimpleFileVisitor<Path>
{
	private static final Logger logger = LoggerFactory.getLogger(TransformFileVisitor.class);
	
	private final Path targetPath;
	private final TemplateTransformer templateTransformer;
	private final TransformerCallback callback;
	private final WebdocContext ctx;
	
	private Path sourcePath = null;

	public TransformFileVisitor(Path targetPath, TemplateTransformer templateTransformer, TransformerCallback callback, WebdocContext ctx)
	{
		this.targetPath = targetPath;
		this.callback = callback;
		this.templateTransformer = templateTransformer;
		this.ctx = ctx;
	}


	@Override
	public FileVisitResult preVisitDirectory(final Path dir, final BasicFileAttributes attrs) throws IOException
	{
		if (sourcePath == null)
			sourcePath = dir;
		else
			Files.createDirectories(targetPath.resolve(sourcePath.relativize(dir)));

		return FileVisitResult.CONTINUE;
	}


	@Override
	public FileVisitResult visitFile(final Path file, final BasicFileAttributes attrs) throws IOException
	{
		//Obtain basedir
		String baseDir = "";

		Path relativePath = sourcePath.relativize(file);
		for (int i=1; i<relativePath.getNameCount(); i++)
			baseDir += "../";

		Path newPath = targetPath.resolve(sourcePath.relativize(file));

		ctx.getContext().put("basedir", baseDir);

		//Mapping found
		File sourceFile = file.toFile();
		byte[] outputData = null;
		
		//update current dir
		ctx.setCurrentDir(sourceFile.getParentFile());
		
		for (RawTransformer t: RawTransformerLoader.getTransformerlist())
		{
			if (file.toString().endsWith(t.getSourceFileExtension() ))
			{
				try(FileInputStream fileInput = new FileInputStream(sourceFile))
				{
					logger.debug("visit file : '{}'", file);
					
					//Get file as string
					outputData = StreamUtil.streamToBytes(fileInput);

					logger.debug("	- read {} bytes", outputData.length);
					
					//Transform with markup transformer
					outputData = t.transform(outputData, ctx);

					logger.debug("	- bytes after markup transformation: {}", outputData.length);
					
					Path relativeFilePath = sourcePath.relativize(file);
					String newFileString = relativeFilePath.toString().substring(0, relativeFilePath.toString().length() - t.getSourceFileExtension().length());
					newFileString += t.getTargetFileExtension();

					//Use extension
					newPath = targetPath.resolve(newFileString);
				}
				catch (Exception e)
				{
					throw new IOException("rawtransform", e);
				}

				break;
			}
		}
			
		for (Transformer t: TransformerLoader.getTransformerlist())
		{
			if (file.toString().endsWith(t.getSourceFileExtension() ))
			{
				try(FileInputStream fileInput = new FileInputStream(sourceFile))
				{
					logger.debug("visit file : '{}'", file);
					
					//Get file as string
					String str = StreamUtil.streamToString(fileInput);

					logger.debug("	- read {} bytes", str.length());
					
					//Apply template engines
					for (TemplateEngine engine: TemplateEngineLoader.getList())
						str = engine.apply(str, ctx);

					logger.debug("	- bytes after templating engine: {}", str.length());
					
					//Transform with markup transformer
					str = t.transform(str, ctx);

					logger.debug("	- bytes after markup transformation: {}", str.length());
					
					//Transform with template transformer
					if (templateTransformer != null)
						str = templateTransformer.applyTemplate(str);
					
					logger.debug("	- bytes after template application: {}", str.length());

					Path relativeFilePath = sourcePath.relativize(file);
					String newFileString = relativeFilePath.toString().substring(0, relativeFilePath.toString().length() - t.getSourceFileExtension().length());
					newFileString += t.getTargetFileExtension();

					//Use extension
					newPath = targetPath.resolve(newFileString);

					outputData = str.getBytes();
				}
				catch (Exception e)
				{
					throw new IOException("transform", e);
				}

				break;
			}
		}

		if (outputData != null)
		{

			callback.callback(file.toString(), newPath.toString());


			File newFile = newPath.toFile();
			newFile.getParentFile().mkdirs();
			FileOutputStream outputStream = new FileOutputStream(newFile);


			outputStream.write(outputData);
			outputStream.close();
		}

		return FileVisitResult.CONTINUE;
	}
}