package io.rudin.tomcat.embedded;

import io.rudin.tomcat.embedded.api.Plugin;
import io.rudin.tomcat.embedded.spi.PluginLoader;

import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;

import li.rudin.core.util.args.ArgumentParser;
import li.rudin.core.util.stream.StreamUtil;
import net.lingala.zip4j.core.ZipFile;

import org.apache.catalina.startup.Tomcat;

public class TomcatStart
{
	
	public static void main(String[] args) throws Exception
	{
		//Invoke init on plugins
		for (Plugin plugin: PluginLoader.getList())
			plugin.init(args);
		
		ArgumentParser parser = new ArgumentParser(args);
		
		if (parser.isPresent("help"))
		{
			usage();
			return;
		}
		
		int port = Integer.parseInt(parser.get("httpPort", "8080"));
		boolean resetExtract = parser.isPresent("resetExtract");
		
		File currentDir = new File(".");
		File baseDir = new File(currentDir, ".tomcat");

		if (resetExtract && baseDir.isDirectory())
			//Delete exploded webapp
			DeleteFileVisitor.delete(baseDir.toPath());
		
		baseDir.mkdir();
		
		File appWar = new File(baseDir, "app.war");
		File appExploded = new File(baseDir, "appExploded");
		
		if (!appExploded.isDirectory())
			extractWebapp(appWar, appExploded);
		
		Tomcat tomcat = new Tomcat();
		
		//Invoke preprocess on plugins
		for (Plugin plugin: PluginLoader.getList())
			plugin.preprocess(tomcat);
		
		tomcat.setBaseDir(baseDir.getAbsolutePath());
		tomcat.setPort(port);
		tomcat.enableNaming();
		
		tomcat.addWebapp("/", appExploded.getAbsolutePath());
		
		//Invoke postprocess on plugins
		for (Plugin plugin: PluginLoader.getList())
			plugin.postprocess(tomcat);

		
		tomcat.start();
		tomcat.getServer().await();
		
		tomcat.stop();
	}
	
	private static void usage()
	{
		System.out.println("Tomcat embedded helper library (https://github.com/rudin-io/tomcat-embedded)");
		System.out.println("");
		System.out.println("Usage:");
		System.out.println("	-httpPort <port>		# use given port");
		System.out.println("	-resetExtract			# removes all previously generated files");
		System.out.println("	-help					# shows this message");
		System.out.println("");
		System.out.println("");
		
		//Invoke printUsage on plugins
		for (Plugin plugin: PluginLoader.getList())
			plugin.printUsage(System.out);

	}
	
	private static void extractWebapp(File appWar, File appExploded) throws Exception
	{
		InputStream inputStream = TomcatStart.class.getResourceAsStream("/app.war");
		
		if (inputStream == null)
			throw new IllegalArgumentException("no embedded war @ /app.war found!");
		
		FileOutputStream outputStream = new FileOutputStream(appWar);
		StreamUtil.copyStream(inputStream, outputStream);
		outputStream.close();
		
		appExploded.mkdir();
		
		ZipFile zipFile = new ZipFile(appWar);
		zipFile.extractAll(appExploded.getAbsolutePath());

	}

}
