package rocketdroid.gradle.dynamic;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URL;
import java.util.Arrays;

public class Repo {
    public static final String ENV_KEY = "ROCKETDROID_REPO";
    public static final String PRODUCTION_REPO = "http://www.rocketdroid.io/repo/";

    public static final long SECONDS = 1000;
    public static final long CHECK_TIME_DEVELOPMENT = 10 * SECONDS;
    public static final long CHECK_TIME_PRODUCTION = 6 * 60 * 60 * SECONDS;

    public static final Repo INSTANCE = new Repo(
            new File(new File(System.getProperty("user.home"), ".rocketdroid"), "libs"),
            getRepo(),
            getCheckTimeInMs());

    public static URI getRepo() {
        try {
            String repoVar = System.getenv(ENV_KEY);
            return repoVar != null ? new URI(repoVar) : new URL(PRODUCTION_REPO).toURI();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private static long getCheckTimeInMs() {
        return System.getenv(ENV_KEY) != null ? CHECK_TIME_DEVELOPMENT : CHECK_TIME_PRODUCTION;
    }

    public final File cache;
    public final URI repo;
    public final long checkTimeInMs;

    public Repo(File cache, URI repo, long checkTimeInMs) {
        this.cache = cache;
        this.repo = repo;
        this.checkTimeInMs = checkTimeInMs;
    }


    public File getArtifact(String artifact) {
        cache.mkdirs();

        File metaFile = new File(cache, artifact + "-metadata.xml");
        File artifactFile = new File(cache, artifact + ".jar");
        try {
            if (updateMetaIfNeeded(artifact, repo, metaFile, checkTimeInMs)) {
                downloadArtifact(artifact, repo, metaFile, artifactFile);
            }
        } catch (IOException e) {
            Log.warn("common/repo", e);
        }

        return artifactFile.exists() ? artifactFile : null;
    }

    /**
     * @return true if updated
     */
    public static boolean updateMetaIfNeeded(String artifact, URI repo, File metaFile, long checkTime) throws IOException {
        Log.info("common/repo", "update meta for " + artifact + " from " + repo + " to " + metaFile);
        if (metaFile.exists() && (System.currentTimeMillis() - metaFile.lastModified() < checkTime)) {
            Log.info("common/repo", "less than " + checkTime + " from last meta update");
            return false;
        }

        byte[] before = metaFile.exists() ? Utils.getBytes(metaFile) : null;
        URI metaUri = repo.resolve("io/rocketdroid/" + artifact + "/1.0-SNAPSHOT/maven-metadata.xml");
        byte[] after = Utils.getUriContent(metaUri);

        if (!Arrays.equals(before, after)) {
            Utils.dump(after, metaFile);
            return true;
        } else {
            metaFile.setLastModified(System.currentTimeMillis());
            return false;
        }
    }

    public static void downloadArtifact(String artifact, URI repo, File metaFile, File artifactFile) throws IOException {
        Log.info("common/repo","download artifact " + artifact + " from " + repo + " to " + artifactFile);

        byte[] metadata = Utils.getBytes(metaFile);
        String timestamp = between(new String(metadata), "<timestamp>", "</timestamp>");
        String buildNumber = between(new String(metadata), "<buildNumber>", "</buildNumber>");

        URI artifactUri = repo.resolve("io/rocketdroid/" + artifact + "/1.0-SNAPSHOT/" + artifact + "-1.0-" + timestamp + "-" + buildNumber + ".jar");
        Utils.dump(Utils.getUriContent(artifactUri), artifactFile);
    }

    private static String between(String content, String before, String after) {
        if (!content.contains(before)) return null;
        content = content.substring(content.indexOf(before) + before.length());
        if (!content.contains(after)) return null;
        content = content.substring(0, content.indexOf(after));
        return content;
    }
}
