package io.relayr.java.storage;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import javax.inject.Inject;
import javax.inject.Singleton;

import io.relayr.java.api.DeviceModelsApi;
import io.relayr.java.model.Device;
import io.relayr.java.model.models.DeviceModel;
import io.relayr.java.model.models.DeviceModels;
import io.relayr.java.model.models.error.DeviceModelsCacheException;
import io.relayr.java.model.models.error.DeviceModelsException;
import rx.Observer;
import rx.Subscriber;

/**
 * Caches all {@link DeviceModel} objects. Works only if there is Internet connection.
 * Use {@link DeviceModelCache} to determine appropriate model for your device.
 * Example: call {@link DeviceModelCache#getModelById(String)} using modelId from {@link Device#getModelId()}
 */
@Singleton
public class DeviceModelCache {

    private static final Map<String, DeviceModel> sDeviceModels = new ConcurrentHashMap<String, DeviceModel>();
    private static volatile boolean refreshing = false;

    private final DeviceModelsApi mModelsApi;

    private int total;
    private int offset;
    private int limit = 5;
    private int timeoutLimit = 0;

    @Inject
    public DeviceModelCache(DeviceModelsApi modelsApi) {
        this.mModelsApi = modelsApi;
        refresh();
    }

    /**
     * Returns cache state. Use this method before using {@link #getModelById(String)}
     * @return true if cache is loading and it's not ready for use, false otherwise
     */
    public boolean isLoading() {
        return refreshing;
    }

    /**
     * Returns {@link DeviceModel} depending on specified modelId.
     * Obtain modelId parameter from {@link Device#getModelId()}
     * @param modelId {@link Device#getModelId()}
     * @return {@link DeviceModel} if one is found, null otherwise
     */
    public DeviceModel getModelById(String modelId) throws DeviceModelsException {
        if (isLoading()) throw DeviceModelsException.cacheNotReady();
        if (modelId == null) throw DeviceModelsException.nullModelId();

        DeviceModel model = sDeviceModels.get(modelId);
        if (model == null) throw DeviceModelsException.deviceModelNotFound();

        return model;
    }

    /**
     * Returns {@link DeviceModel} depending on specified modelId.
     * Search is NOT case sensitive and matches the results with equals.
     * @param name model name
     * @return {@link DeviceModel} if one is found, null otherwise
     */
    public DeviceModel getModelByName(String name) throws DeviceModelsCacheException {
        return getModelByName(name, true, false);
    }

    /**
     * Returns {@link DeviceModel} depending on specified modelId. This search is NOT case sensitive.
     * @param name   model name
     * @param equals if true names will be matched only if they are equal
     * @return {@link DeviceModel} if one is found, null otherwise
     */
    public DeviceModel getModelByName(String name, boolean equals) throws DeviceModelsCacheException {
        return getModelByName(name, equals, false);
    }

    /**
     * Returns {@link DeviceModel} depending on specified modelId.
     * @param name          model name
     * @param equals        if true names will be matched only if they are equal
     * @param caseSensitive true if matching should be case sensitive
     * @return {@link DeviceModel} if one is found, null otherwise
     */
    public DeviceModel getModelByName(String name, boolean equals, boolean caseSensitive) throws DeviceModelsCacheException {
        if (isLoading()) throw DeviceModelsException.cacheNotReady();
        if (name == null || name.trim().isEmpty()) return null;

        String toFind = caseSensitive ? name : name.toLowerCase();
        if (equals)
            for (DeviceModel deviceModel : sDeviceModels.values()) {
                if (!caseSensitive && deviceModel.getName().toLowerCase().equals(toFind))
                    return deviceModel;
                else if (deviceModel.getName().equals(toFind))
                    return deviceModel;
            }
        else
            for (DeviceModel deviceModel : sDeviceModels.values()) {
                if (!caseSensitive && deviceModel.getName().toLowerCase().contains(toFind))
                    return deviceModel;
                else if (deviceModel.getName().contains(toFind))
                    return deviceModel;
            }

        return null;
    }

    /** @return list of all {@link DeviceModel} supported on Relayr platform. */
    public List<DeviceModel> getAll() {
        return new ArrayList<>(sDeviceModels.values());
    }

    /** Refresh device model cache. */
    public void refresh() {
        if (refreshing) return;
        refreshing = true;
        sDeviceModels.clear();

        mModelsApi.getDeviceModels(0, 0)
                .timeout(3, TimeUnit.SECONDS)
                .subscribe(new Subscriber<DeviceModels>() {
                    @Override public void onCompleted() {}

                    @Override public void onError(Throwable e) {
                        refreshing = false;
                        e.printStackTrace();
                        if (e instanceof TimeoutException) {
                            System.out.println("DeviceModuleCache - start loading timeout.");
                            refresh();
                        }
                    }

                    @Override public void onNext(DeviceModels deviceModels) {
                        total = deviceModels.getCount();
                        fetchModels();
                    }
                });
    }

    private void fetchModels() {
        mModelsApi.getDeviceModels(limit, offset)
                .timeout(2, TimeUnit.SECONDS)
                .subscribe(new Observer<DeviceModels>() {
                    @Override public void onCompleted() {}

                    @Override public void onError(Throwable e) {
                        if (e instanceof TimeoutException && timeoutLimit <= 3) {
                            System.out.println("DeviceModuleCache - fetch again due timeout.");
                            timeoutLimit++;
                            fetchModels();
                        } else {
                            if (offset <= total) {
                                offset += limit;
                                fetchModels();
                            } else {
                                refreshing = false;
                            }
                        }
                    }

                    @Override public void onNext(DeviceModels deviceModels) {
                        for (DeviceModel deviceModel : deviceModels.getModels())
                            sDeviceModels.put(deviceModel.getId(), deviceModel);

                        timeoutLimit = 0;
                        offset += limit;
                        if (offset <= total) fetchModels();
                        else refreshing = false;
                    }
                });
    }
}
