package io.relayr.java.model;

import com.google.gson.annotations.SerializedName;

import java.io.Serializable;

import io.relayr.java.RelayrJavaSdk;
import io.relayr.java.model.action.Reading;
import io.relayr.java.model.models.DeviceFirmware;
import io.relayr.java.model.models.DeviceModel;
import io.relayr.java.model.models.error.DeviceModelsException;
import io.relayr.java.model.models.schema.ValueSchema;
import io.relayr.java.model.models.transport.Transport;
import rx.Observable;

/**
 * The Device class is a representation of the device entity.
 * A device entity is any external entity capable of gathering measurements
 * or one which is capable of receiving information from the relayr platform.
 * Examples would be a thermometer, a gyroscope or an infrared sensor.
 */
public class Device implements Serializable {

    private static final long serialVersionUID = 1L;

    private String name;
    private Model model;
    private final String id;
    private final String modelId;
    private final String owner;
    private final String firmwareVersion;
    private final String secret;
    private final String externalId;
    @SerializedName("public") private boolean isPublic;
    @SerializedName("integrationType") private String accountType;

    public Device(String accountType, boolean isPublic, String externalId, String secret,
                  String firmwareVersion, String owner, Model model, String name, String id) {
        this.accountType = accountType;
        this.isPublic = isPublic;
        this.externalId = externalId;
        this.secret = secret;
        this.firmwareVersion = firmwareVersion;
        this.owner = owner;
        this.model = model;
        this.modelId = model.getId();
        this.name = name;
        this.id = id;
    }

    public Device(String id, String owner, String modelId, String name) {
        this.externalId = null;
        this.secret = null;
        this.firmwareVersion = null;
        this.owner = owner;
        this.model = new Model(modelId);
        this.modelId = modelId;
        this.name = name;
        this.id = id;
    }

    /** Used only for Wunderbar devices. */
    public TransmitterDevice toTransmitterDevice() {
        return new TransmitterDevice(id, secret, owner, name, modelId != null ? modelId : model.getId());
    }

    /** Subscribes an app to a device channel. Enables the app to receive data from the device. */
    public Observable<Reading> subscribeToCloudReadings() {
        return RelayrJavaSdk.getWebSocketClient().subscribe(id);
    }

    /** Unsubscribes an app from a device channel, stopping and cleaning up the connection. */
    public void unSubscribeToCloudReadings() {
        RelayrJavaSdk.getWebSocketClient().unSubscribe(id);
    }

    /** Sends a command to the this device */
    public Observable<Void> sendCommand(io.relayr.java.model.action.Command command) {
        return RelayrJavaSdk.getRelayrApi().sendCommand(id, command);
    }

    /**
     * Sends a configuration to device.
     * Check possible configuration in {@link DeviceModel}.
     */
    public Observable<Void> sendConfiguration(io.relayr.java.model.action.Configuration configuration) {
        return RelayrJavaSdk.getRelayrApi().setDeviceConfiguration(id, configuration);
    }

    /**
     * Returns {@link DeviceModel} that defines readings, commands and configurations for
     * specific device depending on device firmware version.
     * Use if {@link RelayrJavaSdk#getDeviceModelsCache()} is initialized.
     * @return {@link DeviceModel}
     */
    public DeviceModel getDeviceModel() throws DeviceModelsException {
        return RelayrJavaSdk.getDeviceModelsCache().getModelById(getModelId());
    }

    /**
     * Returns {@link ValueSchema} for specified meaning and path from received {@link io.relayr.java.model.action.Reading} object.
     * {@link ValueSchema} defines received data and a way to parse it.
     * @return {@link DeviceModel}
     */
    public ValueSchema getValueSchema(String meaning, String path) throws DeviceModelsException {
        DeviceModel model = RelayrJavaSdk.getDeviceModelsCache().getModelById(getModelId());
        if (model == null) throw DeviceModelsException.deviceModelNotFound();

        DeviceFirmware firmware = model.getFirmwareByVersion(firmwareVersion);
        if (firmware == null) throw DeviceModelsException.firmwareNotFound();

        Transport defaultTransport = firmware.getDefaultTransport();
        if (defaultTransport == null) throw DeviceModelsException.transportNotFound();

        return defaultTransport.getReadingByMeaning(meaning, path).getValueSchema();
    }

    public String getModelId() {
        if (modelId != null) return modelId;
        if (model != null) return model.getId();
        return null;
    }

    public String getName() {
        return name;
    }

    public String getId() {
        return id;
    }

    public String getFirmwareVersion() {
        return firmwareVersion;
    }

    public String getSecret() {
        return secret;
    }

    public String getExternalId() {
        return externalId;
    }

    public boolean isPublic() {
        return isPublic;
    }

    public String getOwner() {
        return owner;
    }

    public String getAccountType() {
        return accountType;
    }

    public void setName(String name) {
        this.name = name;
    }

    @Override
    public boolean equals(Object o) {
        return o instanceof TransmitterDevice && ((TransmitterDevice) o).getId().equals(id) ||
                o instanceof Device && ((Device) o).id.equals(id);
    }

    @Override
    public int hashCode() {
        return id.hashCode();
    }

    @Override public String toString() {
        return "Device{" +
                "name='" + name + '\'' +
                ", id='" + id + '\'' +
                ", io.relayr.java.model=" + getModelId() +
                ", owner='" + owner + '\'' +
                ", firmwareVersion='" + firmwareVersion + '\'' +
                ", secret='" + secret + '\'' +
                ", externalId='" + externalId + '\'' +
                ", isPublic=" + isPublic +
                ", accountType='" + accountType + '\'' +
                '}';
    }
}
