package io.relayr.java.api;

import java.util.List;

import io.relayr.java.model.action.Command;
import io.relayr.java.model.action.Configuration;
import io.relayr.java.model.CreateDevice;
import io.relayr.java.model.Device;
import io.relayr.java.model.Transmitter;
import io.relayr.java.model.TransmitterDevice;
import io.relayr.java.model.models.schema.ValueSchema;
import io.relayr.java.model.models.transport.DeviceCommand;
import io.relayr.java.model.models.transport.DeviceConfiguration;
import io.relayr.java.model.onboarding.OnBoardingState;
import retrofit.client.Response;
import retrofit.http.Body;
import retrofit.http.DELETE;
import retrofit.http.GET;
import retrofit.http.PATCH;
import retrofit.http.POST;
import retrofit.http.Path;
import retrofit.http.Query;
import retrofit.http.Streaming;
import rx.Observable;

/** This class incorporates a wrapped version of the relayr API calls. */
public interface RelayrApi {

    /**
     * Creates device on the platform.
     * @param device to create
     * @return created device
     */
    @POST("/devices") Observable<Device> createDevice(@Body CreateDevice device);

    /**
     * Send command defined by device models {@link DeviceCommand}
     * Before sending a command make sure to validate it by calling {@link Command#validate(ValueSchema)}
     */
    @POST("/devices/{device_id}/cmd")
    Observable<Void> sendCommand(@Path("device_id") String deviceId,
                                 @Body Command command);

    /**
     * Deletes a device from platform.
     * @param deviceId id of device to delete
     * @return 200 OK if successful
     */
    @DELETE("/devices/{device_id}")
    Observable<Void> deleteDevice(@Path("device_id") String deviceId);

    /**
     * Updates a device.
     * @param device   updated device with the new details
     * @param deviceId id of the device to update
     * @return an {@link Observable} to the updated Device
     */
    @PATCH("/devices/{deviceId}") Observable<Device> updateDevice(@Body Device device,
                                                                  @Path("deviceId") String deviceId);

    /**
     * A public device is a device which public attribute has been set to 'true' therefore
     * no authorization is required.
     * @param meaning When a meaning is specified, the request returns only
     *                the devices which readings match the meaning.
     * @return an {@link Observable} with a list of all public devices.
     */
    @GET("/devices/public")
    Observable<List<Device>> getPublicDevices(@Query("meaning") String meaning);

    /**
     * @return an {@link Observable} of a specific transmitter
     */
    @GET("/transmitters/{transmitter}")
    Observable<Transmitter> getTransmitter(@Path("transmitter") String transmitter);

    /**
     * Updates a transmitter.
     * @param transmitter   updated transmitter with the new details
     * @param transmitterId id of the transmitter to update
     * @return an {@link Observable} to the updated Transmitter
     */
    @PATCH("/transmitters/{transmitter}")
    Observable<Transmitter> updateTransmitter(@Body Transmitter transmitter,
                                              @Path("transmitter") String transmitterId);

    /**
     * @param transmitter the id of the transmitter to get the devices from
     * @return an {@link Observable} with a list of devices that belong to the specific
     * transmitter.
     */
    @GET("/transmitters/{transmitter}/devices")
    Observable<List<TransmitterDevice>> getTransmitterDevices(
            @Path("transmitter") String transmitter);

    /**
     * Registers the transmitter
     * @param transmitter transmitter object to register
     * @return an {@link Observable} to the registered Transmitter
     */
    @POST("/transmitters")
    Observable<Transmitter> registerTransmitter(@Body Transmitter transmitter);

    /**
     * Deletes a transmitter and all of its components (Transmitter and Devices)
     * @param transmitterId id of the transmitter (the Master Module)
     * @return an empty {@link Observable}
     */
    @DELETE("/transmitters/{transmitterId}")
    Observable<Void> deleteTransmitter(@Path("transmitterId") String transmitterId);

    /**
     * Returns true if transmitter is connected to MQTT and able to send data.
     * @param transmitterId id of the transmitter (the Master Module)
     * @return an empty {@link Observable}
     */
    @GET("/experimental/transmitters/{transmitterId}/state")
    Observable<OnBoardingState> isTransmitterConnected(@Path("transmitterId") String transmitterId);

    /**
     * Returns true if device is connected to MQTT and able to send data.
     * @param deviceId id of the device {@link Device#id}
     * @return an empty {@link Observable}
     */
    @GET("/experimental/devices/{deviceId}/state")
    Observable<OnBoardingState> isDeviceConnected(@Path("deviceId") String deviceId);

    /**
     * Returns true if transmitter is connected to MQTT and able to send data.
     * @param transmitterId id of the transmitter (the Master Module)
     * @return an empty {@link Observable}
     */
    @POST("/experimental/transmitters/{transmitterId}/ble-scan/{period}")
    @Streaming Observable<Response> scanForDevices(@Path("transmitterId") String transmitterId,
                                                   @Path("period") int period);

    /**
     * Deletes a WunderBar and all of its components (Transmitter and Devices)
     * @param transmitterId id of the transmitter (the Master Module)
     * @return an empty {@link Observable}
     */
    @DELETE("/wunderbars/{transmitterId}")
    Observable<Void> deleteWunderBar(@Path("transmitterId") String transmitterId);

    /**
     * Returns current device configuration.
     * @param path path defined in {@link DeviceConfiguration#getPath()}
     * @return an existing configuration {@link Observable}
     */
    @GET("/devices/{deviceId}/configuration")
    Observable<Configuration> getDeviceConfiguration(@Path("deviceId") String deviceId,
                                                     @Query("path") String path);

    /**
     * Sets device configuration defined by device models {@link DeviceConfiguration}
     * Before sending a command make sure to validate it by calling {@link Configuration#validate(ValueSchema)}
     */
    @POST("/devices/{deviceId}/configuration")
    Observable<Void> setDeviceConfiguration(@Path("deviceId") String deviceId,
                                            @Body Configuration configuration);
}
