package io.relayr.java.storage;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import javax.inject.Inject;
import javax.inject.Singleton;

import io.relayr.java.api.DeviceModelsApi;
import io.relayr.java.model.Device;
import io.relayr.java.model.models.DeviceModel;
import io.relayr.java.model.models.DeviceModels;
import io.relayr.java.model.models.error.DeviceModelsException;
import rx.Subscriber;

/**
 * Caches all {@link DeviceModel} objects. Works only if there is Internet connection.
 * Use {@link DeviceModelCache} to determine appropriate model for your device.
 * Example: call {@link DeviceModelCache#getModel(String)} using modelId from {@link Device#getModelId()}
 */
@Singleton
public class DeviceModelCache {

    private static final Map<String, DeviceModel> sDeviceModels = new ConcurrentHashMap<String, DeviceModel>();
    private static volatile boolean refreshing = false;

    private final DeviceModelsApi mModelsApi;

    @Inject
    public DeviceModelCache(DeviceModelsApi modelsApi) {
        this.mModelsApi = modelsApi;
        refresh();
    }

    /**
     * Returns cache state. Use this method before using {@link #getModel(String)}
     * @return true if cache is ready false otherwise
     */
    public boolean isEmpty() {
        return refreshing || sDeviceModels.isEmpty();
    }

    /**
     * Returns {@link DeviceModel} depending on specified modelId.
     * Obtain modelId parameter from {@link Device#getModelId()}
     * @param modelId {@link Device#getModelId()}
     * @return {@link DeviceModel} if one is found, null otherwise
     */
    public DeviceModel getModel(String modelId) throws DeviceModelsException {
        if (isEmpty()) throw DeviceModelsException.cacheNotReady();
        if (modelId == null) throw DeviceModelsException.nullModelId();

        DeviceModel model = sDeviceModels.get(modelId);
        if (model == null) throw DeviceModelsException.deviceModelNotFound();

        return model;
    }

    /** @return list of all {@link DeviceModel} supported on Relayr platform. */
    public List<DeviceModel> getAll() {
        return new ArrayList<>(sDeviceModels.values());
    }

    /** Refresh device model cache. */
    public void refresh() {
        if (refreshing) return;
        refreshing = true;

        mModelsApi.getDeviceModels(50)
                .timeout(7, TimeUnit.SECONDS)
                .subscribe(new Subscriber<DeviceModels>() {
                    @Override
                    public void onCompleted() {
                        refreshing = false;
                    }

                    @Override
                    public void onError(Throwable e) {
                        refreshing = false;
                        e.printStackTrace();
                        if (e instanceof TimeoutException) refresh();
                    }

                    @Override
                    public void onNext(DeviceModels deviceModels) {
                        for (DeviceModel deviceModel : deviceModels.getModels())
                            sDeviceModels.put(deviceModel.getId(), deviceModel);
                        refreshing = false;
                    }
                });
    }
}
