package io.relayr.java;

import javax.inject.Inject;

import io.relayr.java.api.AccountsApi;
import io.relayr.java.api.DeviceModelsApi;
import io.relayr.java.api.GroupsApi;
import io.relayr.java.api.RelayrApi;
import io.relayr.java.api.UserApi;
import io.relayr.java.model.Device;
import io.relayr.java.model.Transmitter;
import io.relayr.java.model.User;
import io.relayr.java.model.account.Account;
import io.relayr.java.model.groups.Group;
import io.relayr.java.model.models.DeviceModel;
import io.relayr.java.storage.DeviceModelCache;
import io.relayr.java.websocket.WebSocketClient;
import rx.Observable;

/**
 * The RelayrJavaSdk Class serves as the access point to all endpoints in the Android SDK.
 * It includes basic calls such as user login validation and can also call the handlers of the
 * -----------------------------------------------------------------------------------------------
 * For easy start, after logging in, obtain {@link User} by calling {@link #getUser()}.
 * This is main object in Relayr SDK that can be used to fetch users {@link Device},
 * {@link Transmitter}, {@link Group} and {@link Account} objects.
 * Every mentioned object has it's own interaction methods so direct usage of APIs is not necessary.
 * However it's still possible to obtain any of the desired API handlers through appropriate method:
 * <ul>
 * <li>{@link #getUserApi()}</li>
 * <li>{@link #getRelayrApi()}</li>
 * <li>{@link #getGroupsApi()}</li>
 * <li>{@link #getAccountsApi()}</li>
 * <li>{@link #getDeviceModelsApi()} or using cache {@link #getDeviceModelsCache()}</li>
 * </ul>
 * For other details check methods JavaDoc
 */
public class RelayrJavaSdk {

    @Inject static UserApi mUserApi;
    @Inject static RelayrApi mRelayrApi;

    @Inject static GroupsApi mGroupsApi;
    @Inject static AccountsApi mAccountsApi;
    @Inject static DeviceModelsApi sDeviceModelsApi;
    @Inject static DeviceModelCache sDeviceModelsCache;

    @Inject static WebSocketClient mWebSocketClient;

    /** Initializes the SDK. */
    public static class Builder {

        private boolean inMock;
        private String token;

        public Builder() {
        }

        /**
         * Initializes the SDK in Mock Mode.
         * In this mode, mock reading values are generated.
         * Used for testing purposes, without the need of a WunderBar or an internet connection.
         */
        public Builder inMockMode(boolean mockMode) {
            this.inMock = mockMode;
            return this;
        }

        /**
         * Initializes the SDK in Mock Mode.
         * In this mode, mock reading values are generated.
         * Used for testing purposes, without the need of a WunderBar or an internet connection.
         */
        public Builder setToken(String token) {
            this.token = token;
            return this;
        }

        public void build() {
            RelayrJavaApp.init(token, inMock);
        }
    }

    /**
     * Returns user object. This is main object in Relayr SDK that can be used to fetch user {@link Device},
     * {@link Transmitter}, {@link Group} and {@link Account}.
     * Any of the mentioned objects has it's own special methods so direct usage of APIs is not necessary but it's still possible
     */
    public static Observable<User> getUser() {
        return mUserApi.getUserInfo();
    }

    /**
     * Returns the version of the SDK
     * @return the version String
     */
    public static String getVersion() {
        return "";
    }

    /**
     * Returns the handler of the Relayr API. Use after obtaining User data.
     * Used as an access point to class {@link RelayrApi}
     */
    public static RelayrApi getRelayrApi() {
        return mRelayrApi;
    }

    /**
     * Returns the handler of the Accounts API. Use to add third party accounts to the relayr user.
     * Used as an access point to class {@link AccountsApi}
     */
    public static AccountsApi getAccountsApi() {
        return mAccountsApi;
    }

    /**
     * Returns the handler of the Groups API.
     * Used as an access point to class {@link GroupsApi}
     */
    public static GroupsApi getGroupsApi() {
        return mGroupsApi;
    }

    /**
     * Return the handler of the relayr DeviceModels API. Returns new device models. To use properly
     * Used as an access point to class {@link DeviceModelsApi}
     */
    public static DeviceModelsApi getDeviceModelsApi() {
        return sDeviceModelsApi;
    }

    /**
     * Returns cached {@link DeviceModel} objects.
     * Cache will be populated with models from {@link Device#model} when device is fetched.
     * Use instead of {@link #getDeviceModelsApi()}
     */
    public static DeviceModelCache getDeviceModelsCache() {
        return sDeviceModelsCache;
    }

    /**
     * @return the handler of the relayr User API. Use to get user data and user
     * transmitters, devices and accounts
     * Used as an access point to class {@link UserApi}
     */
    public static UserApi getUserApi() {
        return mUserApi;
    }

    /**
     * Used as an access point to the class {@link io.relayr.java.websocket.WebSocketClient}
     * @return the handler of the WebSocket client
     */
    public static WebSocketClient getWebSocketClient() {
        return mWebSocketClient;
    }
}
